# -*- coding: utf-8 -*-

import distutils.dir_util
import os.path
import logging

from sandbox.common.types import client
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.nanny import nanny as common_nanny
from sandbox.projects.userdata import resources as userdata_resources
from sandbox.projects import resource_types


class UserfeatPrepareBinariesRelease(common_nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Merge contents of main Debian packages of User Features to deploy via Nanny
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = client.Tag.LINUX_PRECISE
        ram = 2 << 10  # in MiB, 2 GiB
        disk_space = 80 << 10  # in MiB, 80 GiB

    class Parameters(sdk2.Task.Parameters):
        # Transmit subversion URL to ReleaseSearchComponent2
        with sdk2.parameters.Group(build_params.BASE_BUILD_GROUP_NAME) as bb_group:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        with sdk2.parameters.Group("Input parameters") as input_parameters_group:
            miscdata_package_resid = sdk2.parameters.Resource(
                "Debian package of misc data pipeline",
                required=False,
                resource_type=resource_types.USERFEAT_MISCDATA_PACKAGE,
            )
            user_browse_package_resid = sdk2.parameters.Resource(
                "Debian package of user browse pipeline",
                required=False,
                resource_type=resource_types.USERFEAT_USER_BROWSE_PACKAGE,
            )
            user_counters_package_resid = sdk2.parameters.Resource(
                "Debian package of user counters pipeline",
                required=False,
                resource_type=resource_types.USERFEAT_USER_COUNTERS_PACKAGE,
            )
            user_search_package_resid = sdk2.parameters.Resource(
                "Debian package of user search pipeline",
                required=False,
                resource_type=resource_types.USERFEAT_USER_SEARCH_PACKAGE,
            )
            userdata_package_resid = sdk2.parameters.Resource(
                "Debian package of user data pipeline",
                required=False,
                resource_type=resource_types.USERFEAT_USERDATA_PACKAGE,
            )
        with sdk2.parameters.Group("Output parameters") as output_parameters_group:
            result_description = sdk2.parameters.String(
                label="Result description",
                default_value="<no description>",
            )

    class Context(sdk2.Task.Context):
        pass

    def extract_content(self, src, dst):
        logging.info("Extracting content of %s...", src.type)
        workdir = "extraction"
        if os.path.exists(workdir):
            logging.info("Removing previously created dir %s", workdir)
            distutils.dir_util.remove_tree(workdir)
        os.makedirs(workdir)
        src_data = sdk2.ResourceData(src)
        dst_data = sdk2.ResourceData(dst)
        resource_tar_path = src_data.path
        content_path = dst_data.path
        content_path.mkdir(0o755, parents=True, exist_ok=True)
        log_name = "extraction_{}".format(resource_tar_path.name)
        with sdk2.helpers.ProcessLog(self, logger=log_name) as pl:
            pl.logger.propagate = 1
            get_deb_command = "tar xvf \"{resource_tar_path}\" -C \"{workdir}\"".format(
                resource_tar_path=resource_tar_path,
                workdir=workdir,
            )
            sp.Popen(get_deb_command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()
            deb_files = [path for path in os.listdir(workdir) if path.endswith(".deb")]
            assert len(deb_files) == 1
            deb_path = os.path.join(workdir, deb_files[0])
            unpack_deb_command = "dpkg-deb -x \"{deb_path}\" \"{content_path}\"".format(
                deb_path=deb_path,
                content_path=content_path,
            )
            sp.Popen(unpack_deb_command, shell=True, stdout=pl.stdout, stderr=sp.STDOUT).wait()

    def on_execute(self):
        source_resources = [
            (self.Parameters.miscdata_package_resid, userdata_resources.UserfeatMiscBinariesRelease),
            (self.Parameters.user_browse_package_resid, userdata_resources.UserfeatBrowseBinariesRelease),
            (self.Parameters.user_counters_package_resid, userdata_resources.UserfeatCountersBinariesRelease),
            (self.Parameters.user_search_package_resid, userdata_resources.UserfeatSearchBinariesRelease),
            (self.Parameters.userdata_package_resid, userdata_resources.UserfeatUserdataBinariesRelease),
        ]
        source_resources = [(resid, output_type) for resid, output_type in source_resources if resid]
        assert source_resources, "At least one resource must be provided"
        source_resource_ids = []
        for resid, _ in source_resources:
            source_resource_ids.append("{}={}".format(resid.type, resid.id))

        output_type = userdata_resources.UserfeatBinariesRelease if len(source_resources) > 1 else source_resources[0][1]

        binaries_release = output_type(
            task=self,
            description=self.Parameters.result_description,
            path="userfeat_binaries_release",
            source_resource_ids=",".join(source_resource_ids),
        )

        for r in source_resources:
            self.extract_content(r[0], binaries_release)
