# -*- coding: utf-8 -*-

import datetime
import distutils.dir_util
import distutils.file_util
import logging

from sandbox.common.types import client
from sandbox.common.types import resource as sandbox_resource
from sandbox import sdk2
from sandbox.projects.common.nanny import nanny as common_nanny
from sandbox.projects.common.userdata import resource_types as common_userdata_resources
from sandbox.projects.userdata import resources as userdata_resources
from sandbox.projects.geobase import Geodata5BinStable
from sandbox.projects.logs import resources as us_resources
from sandbox.projects import ipreg as ipreg_resources
from sandbox.projects.sandbox import remote_copy_resource, remote_copy_resource_2, http_upload_2
from sandbox.projects import resource_types
from sandbox.projects import ClickdaemonKeygen


class UserfeatPrepareDataRelease(sdk2.Task):
    """
    Collect data resources from different sources and pack for release
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = client.Tag.LINUX_PRECISE
        required_ram = 2 << 10
        execution_space = 20000

    class Parameters(sdk2.Task.Parameters):
        result_description = sdk2.parameters.String(
            label="Result description",
            default_value="<no description>",
        )
        result_released_ttl = sdk2.parameters.String(
            label="Result resource ttl after release",
            default_value="inf",
        )

    class Context(sdk2.Task.Context):
        pass

    def find_resource(self, task_type, resource_type, attrs, max_age_days):
        items = sdk2.Resource[resource_type].find(
            state=sandbox_resource.State.READY,
            attrs=attrs,
        ).order(-sdk2.Resource.id).limit(100)
        for item in items:
            if task_type is None or str(item.task.type) == str(task_type):
                today = datetime.datetime.today().replace(tzinfo=None)
                created = item.created.replace(tzinfo=None)
                assert today - created < datetime.timedelta(days=max_age_days)
                return item
        raise Exception("unable to find any resource {} by {}, attrs {}".format(
            resource_type, task_type, attrs
        ))

    def collect_resources(self):
        resources = []
        released_to_stable = {sandbox_resource.ServiceAttributes.RELEASED: "stable"}
        # Created by regular task https://sandbox.yandex-team.ru/scheduler/9372/view
        resources.append(self.find_resource(
                task_type=Geodata5BinStable.GEODATA5BIN_STABLE_BUILDER,
                resource_type=Geodata5BinStable.GEODATA5BIN_STABLE,
                attrs=released_to_stable,
                max_age_days=7,
        ))
        # Exported to sandbox regularly: https://st.yandex-team.ru/IPREG-2785
        resources.append(self.find_resource(
                task_type=http_upload_2.HTTPUpload2.type,
                resource_type=ipreg_resources.IPREG_EXPORT,
                attrs=released_to_stable,
                max_age_days=7,
        ))
        # The resource was being copied by regular task https://sandbox.yandex-team.ru/scheduler/1055/view
        # The source broke in May 2020, we use the last retrieved resource from now on.
        resources.append(self.find_resource(
                task_type=remote_copy_resource.RemoteCopyResource.type,
                resource_type=resource_types.GEOA_C2P,
                attrs={
                    "userdata_release": "yes",
                },
                max_age_days=1830,
        ))
        # Created by regular task https://sandbox.yandex-team.ru/scheduler/1116/view
        resources.append(self.find_resource(
                task_type=remote_copy_resource_2.RemoteCopyResource2.type,
                resource_type=us_resources.SESSIONS_BLOCKSTAT,
                attrs={
                    "user_search_release": "yes",
                },
                max_age_days=7,
        ))
        # Created by regular task https://sandbox.yandex-team.ru/scheduler/729/view
        resources.append(self.find_resource(
                task_type=ClickdaemonKeygen.ClickdaemonKeygen.type,
                resource_type=resource_types.CLICKDAEMON_KEYS,
                attrs={},
                max_age_days=7,
        ))
        return resources

    def copy_resource_data(self, src, dst):
        """
        :param src: sdk2.Resource
        :param dst: sdk2.Resource
        """
        src_data = sdk2.ResourceData(src)
        dst_data = sdk2.ResourceData(dst)
        src_path = src_data.path
        dst_dirpath = dst_data.path
        dst_dirpath.mkdir(0o755, parents=True, exist_ok=True)
        dst_path = dst_dirpath / src_path.name
        logging.info("copying from {} to {}".format(src_path, dst_path))
        if src_path.is_file():
            distutils.file_util.copy_file(str(src_path), str(dst_path))
        else:
            distutils.dir_util.copy_tree(str(src_path), str(dst_path))

    def on_execute(self):
        resources = self.collect_resources()
        source_resource_ids = []
        for r in resources:
            source_resource_ids.append("{}={}".format(r.type, r.id))
        data_release = userdata_resources.UserfeatDataRelease(
            task=self,
            description=self.Parameters.result_description,
            path="userfeat_data_release",
            arch="any",
            source_resource_ids=",".join(source_resource_ids),
        )
        for r in resources:
            logging.info("Collecting resource %d", r.id)
            self.copy_resource_data(r, data_release)

    def on_release(self, additional_parameters):
        released_ttl = self.Parameters.result_released_ttl
        if not (released_ttl == "inf" or released_ttl.isdigit()):
            raise ValueError("Resource TTL must be 'inf' or integer")
        self.mark_released_resources(
            additional_parameters["release_status"],
            ttl=released_ttl,
        )
