# -*- coding: utf-8 -*-

import json
import logging
import time

import sandbox.projects.release_machine.core.task_env as task_env

from sandbox import common
from sandbox import sdk2

from sandbox.common.types import task as sandbox_task
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.release_machine.helpers import startrek_helper
from sandbox.projects.common.sdk_compat import task_helper as rm_task
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.userfeat import UserfeatCfg
from sandbox.sandboxsdk.environments import PipEnvironment

USERFEAT_COMPONENT_NAME = UserfeatCfg.name


class DeploymentPolicy(object):
    REGULAR = "regular"
    FORCED = "forced"


class UserfeatPrepareForDeploy(sdk2.Task):
    """
    The tasks prepares userfeat components for deploy.
    - Marks passed resources as released. This is required for release tests, so they can find production version.
    - Sets YT attribute with resource ids, so it can be used from ClusterMaster deploy graph or by other method.

    It doesn't initiate deploy on nanny.
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            task_env.TaskRequirements.startrek_client,
            PipEnvironment('yandex-yt', use_wheel=True),
            PipEnvironment('yandex-yt-yson-bindings-skynet', use_wheel=True)
        ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("Roll packages to") as roll_to:
            roll_to.values[rm_const.ReleaseStatus.stable] = roll_to.Value(
                value=rm_const.ReleaseStatus.stable,
                default=True
            )
            roll_to.values[rm_const.ReleaseStatus.testing] = roll_to.Value(value=rm_const.ReleaseStatus.testing)
            roll_to.values[rm_const.ReleaseStatus.prestable] = roll_to.Value(value=rm_const.ReleaseStatus.prestable)
            roll_to.values[rm_const.ReleaseStatus.unstable] = roll_to.Value(value=rm_const.ReleaseStatus.unstable)

        resources = sdk2.parameters.Resource(
            label="Resources to be released",
            required=True,
            multiple=True,
            description="All the resources to be released",
        )

        with sdk2.parameters.RadioGroup(label="Deployment policy",
                                        description="Wait till ClusterMaster graphs finish (regular) or not (forced)") as policy:
            policy.values[DeploymentPolicy.REGULAR] = policy.Value(value=DeploymentPolicy.REGULAR, default=True)
            policy.values[DeploymentPolicy.FORCED] = policy.Value(value=DeploymentPolicy.FORCED)

        release_number = sdk2.parameters.Integer(
            label="Release number",
            required=True,
            description="Major release number (branch number)",
        )

        revision_number = sdk2.parameters.Integer(
            label="Revision number",
            required=True,
            description="Revision number",
        )
        minor_release_number = sdk2.parameters.Integer(
            label="Minor release number",
            required=True,
            description="Minor release number (tag number)"
        )

        yt_cluster = sdk2.parameters.String(
            label="YT cluster",
            required=False,
            description="YT cluster",
        )

        yt_attribute_path = sdk2.parameters.String(
            label="YT attribute path",
            required=False,
            description="YT attribute where release info should be stored",
        )

        vault_yt_token = sdk2.parameters.String(
            label="YT token vault item name",
            required=False,
            description="YT token vault item name",
        )

        mark_resources_as_released = sdk2.parameters.Bool(
            label="Mark resources as released",
            required=False,
            description="Set True if you need to mark resources as released in sandbox",
            default=True,
        )

        need_send_st = sdk2.parameters.Bool(
            label="Need to send st message",
            required=False,
            description="Need to send message to st ticket",
            default=True,
        )

    def on_execute(self):
        info = self.collect_deploy_info()
        tasks = set([res_info["task_id"] for res_info in info["resources"].itervalues()])

        if (self.Parameters.yt_cluster and self.Parameters.vault_yt_token and self.Parameters.yt_attribute_path):
            self.set_yt_attr(info)
        else:
            assert not (self.Parameters.yt_cluster or self.Parameters.vault_yt_token or self.Parameters.yt_attribute_path), "All YT-related parameters should be set or unset simultaneously"
            logging.info("YT-related parameters were not passed, not setting YT attrbiute")

        if self.Parameters.mark_resources_as_released:
            self.release_tasks(tasks)

        if self.Parameters.mark_resources_as_released and not self.wait_for_released(tasks):
                self.Context.fail_msg = ("Failed to mark tasks as released to {}".format(self.Parameters.roll_to) +
                                         " after successful YT attribute update. \n" +
                                         "Please, check the Sandbox tasks via web interface!")
                eh.check_failed(self.Context.fail_msg)

        if self.Parameters.need_send_st:
            comment = "The following tasks have been released in {}:\n{}".format(
                common.utils.get_task_link(self.id),
                "".join(["- {}\n".format(common.utils.get_task_link(task_id)) for task_id in tasks])
            )
            self.post_to_st_issue(comment)

    def collect_deploy_info(self):
        info = {
            "policy": self.Parameters.policy,
            "release": self.Parameters.release_number,
            "minor_release_number": self.Parameters.minor_release_number,
            "revision": self.Parameters.revision_number,
            "resources": {}
        }

        for res in self.Parameters.resources:
            info["resources"][str(res.type)] = {
                "resource_id": int(res.id),
                "task_id": int(res.task_id)
            }

        logging.info("Got resources %s", json.dumps(info, indent=4))

        return info

    def set_yt_attr(self, yt_attr):
        import yt.wrapper as yt
        yt_client = yt.YtClient(proxy=self.Parameters.yt_cluster, token=sdk2.Vault.data(self.Parameters.vault_yt_token))
        yt_client.set(self.Parameters.yt_attribute_path, yt_attr)

    def post_to_st_issue(self, text):
        c_info = rmc.COMPONENTS[USERFEAT_COMPONENT_NAME]()
        st_auth_token = sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME)
        st_helper = startrek_helper.STHelper(st_auth_token)
        st_helper.comment(self.Parameters.release_number, text, c_info)

    def release_tasks(self, tasks):
        api = common.rest.Client()
        roll_to = self.Parameters.roll_to
        for task_id in tasks:
            logging.info("Releasing {} to {}...".format(task_id, roll_to))
            payload = {
                "task_id": task_id,
                "cc": [],
                "to": [],
                "params": {},
                "message": "",
                "type": roll_to,
                "subject": "",
            }
            logging.debug("API request 'release' with body: {}".format(payload))
            self.release_single_task(api, payload)

    @staticmethod
    @decorators.retries(max_tries=3, delay=5, raise_class=common.errors.TaskError)
    def release_single_task(api, payload):
        api.release(payload)

    def wait_for_released(self, tasks):
        start_time = int(time.time())
        # wait for build task released status up to 30 minutes
        while int(time.time()) - start_time < 30 * 60:
            if self.check_released_tasks(tasks):
                return True
            time.sleep(15)
        return False

    def check_released_tasks(self, tasks):
        """
        :return: True if all tasks got "RELEASED", False otherwise
        """
        not_released = 0
        for task_id in tasks:
            task_status = rm_task.task_status(task_id)
            logging.info("task #{} has status {}".format(task_id, task_status))
            if task_status != sandbox_task.Status.RELEASED:
                not_released += 1
        return not_released == 0
