# -*- coding: utf-8 -*-

import logging
import tempfile

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

import sandbox.projects.verticals.common.saas_tools as saas_tools
from generate_talents_data import TTalentsDataGenerator

logger = logging.getLogger(__name__)


TALENTS_DATA_YT_SERVER = "hahn.yt.yandex.net"
TALENTS_EXPORT_PATH = "//home/talents/export/wizard"
TALENTS_KV_SAAS_TABLE = "//home/peoplesearch/talents/saas-state"
TALENTS_SAAS_NAMESPACE = "yandex_talents"


class YaTalentsUpdateSnippets(sdk2.Task):
    """Regenerate data for Yandex Talents snippets in Saas"""

    class Parameters(sdk2.Task.Parameters):
        yt_token_vault = sdk2.parameters.String(
            "YT_TOKEN name in vault",
            required=True,
        )

        show_extended_params = sdk2.parameters.Bool(
            "Show extended params",
            default=False,
        )

        with show_extended_params.value[True]:
            saas_url = sdk2.parameters.String(
                "Saas upload url",
                default=saas_tools.KV_SAAS_UPPER_UPLOAD_URL,
            )
            saas_namespace = sdk2.parameters.String(
                "Saas namespace",
                default=TALENTS_SAAS_NAMESPACE,
            )
            talents_export_table = sdk2.parameters.String(
                "Talents export table",
                default=TALENTS_KV_SAAS_TABLE,
            )

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    def _update_saas_snippets(self):
        with tempfile.NamedTemporaryFile(bufsize=0) as token_file:
            token_file.write(sdk2.Vault.data(self.Parameters.yt_token_vault))

            generator = TTalentsDataGenerator(TALENTS_DATA_YT_SERVER, TALENTS_EXPORT_PATH, token_file.name)
            generator.generate_saas_data("saas.data")

            saas_tools.upload_trie_to_yt_table(
                input_file="saas.data",
                yt_token_file=token_file.name,
                yt_output_table=TALENTS_KV_SAAS_TABLE,
            )
            saas_tools.upload_trie_to_saas(
                saas_url=self.Parameters.saas_url,
                yt_token_file=token_file.name,
                yt_input_table=self.Parameters.talents_export_table,
                namespace=self.Parameters.saas_namespace,
                namespace_suffix='""',
            )

    def on_execute(self):
        with self.memoize_stage.update_saas_snippets(commit_on_entrance=False):
            self._update_saas_snippets()
