from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox.projects import resource_types
import sys
import sandbox.sandboxsdk.svn as sdk_svn

import time
import os


class SelfServiceAutotest(sdk2.Task):
    """
        Build self-service-api and create test database
    """

    class Context(sdk2.Task.Context):
        build_task_id = None
        test_schema = ""
        hash = ""

    class Parameters(sdk2.Parameters):
        schema = sdk2.parameters.String(
            "schema",
            default="vh_self_service_test",
        )

        revision = sdk2.parameters.Integer(
            "revision",
            default=None
        )

        arcadia_path = sdk2.parameters.String(
            "arcadia_path",
            default="arcadia:/arc/trunk/arcadia"
        )

    def create_build_subtask(self):
        task = sdk2.Task["YA_MAKE"]

        checkout_arcadia_from_url = self.Parameters.arcadia_path
        revision = self.Parameters.revision
        checkout_arcadia_from_url += "@{}".format(revision) if revision else ""

        return task(
            self,
            description="build self-service-api",
            owner=self.owner,
            result_rt="OTHER_RESOURCE",
            checkout_arcadia_from_url=checkout_arcadia_from_url,
            targets="yabs/vh/cms-pgaas/self_service_api",
            arts="yabs/vh/cms-pgaas/self_service_api/self-service-api",
            result_single_file=True
        )

    def _set_workflow(self, workflow):
        self.workflow = workflow
        self.workflow_id = self.workflow.wfId

    def on_execute(self):
        with self.memoize_stage.create_db:
            sys.path.append(sdk_svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/tools/mstand/nirvana_api"))

            self.create_database()

        with self.memoize_stage.create_build_task:
            buld_task = self.create_build_subtask()
            self.Context.build_task_id = buld_task.id
            buld_task.enqueue()

            raise sdk2.WaitTask(
                [buld_task],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        task = sdk2.Task.find(
            id=self.Context.build_task_id,
            children=True
        ).first()

        if task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskError("Build task failed")

    def _set_param(self, block, param, value):
        self.workflow.SetBlockParametersByGuid(block["blockGuid"], param, value)

    def create_database(self):
        import nirvana_api as n_api
        schema = self.Parameters.schema + "_" + str(int(time.time()))
        self.set_info('test schema {}'.format(schema))

        cmd = """/self-service-api \
        --connection-string "host=pgaas-test.mail.yandex.net port=12000 sslmode=require dbname=vh_testenv \
        user=testenv password=$VH_TESTENV_PG_PASSWORD" -p 80 --config-file-path /config --schema {schema} \
        --disable-boto 1 --log-flavor qloud --use-test-blackbox-tvm yes \
        --stats-table VideohostingViewsDetailedStatAssessors""".format(schema=schema)
        resource = sdk2.ResourceData(resource_types.OTHER_RESOURCE(
            self, "Output file", "filename.txt"
        ))
        resource.path.write_bytes(cmd)
        os.chmod(str(resource.path), 0777)
        resource.ready()

        nirvana_token = sdk2.Vault.data("VIDEO-BASE-DEPLOY", "robot_video_acc_nirvana_token")
        workflow_id = "facd8d5a-bb5e-466a-a72c-019a751bd89c"
        self._set_workflow(n_api.NirvanaAPI(workflow_id, nirvana_token))
        self.set_info('workflow {}'.format(self.workflow_id))

        self._set_workflow(self.workflow.CloneThisWorkflow())
        block = self.workflow.GetAllBlockParameters()["result"][0]
        self._set_param(block, "schema", schema)
        self.workflow.StartWorkflow()

        completion_status = self.wait_launch_complete()

        if completion_status != "success":
            raise common.errors.TaskError("Workflow failed")

    def get_workflow_status(self):
        state = self.workflow.GetExecutionState()["result"]

        return {
            "status": state["status"],
            "result": state["result"]
        }

    def wait_launch_complete(self):
        info = self.get_workflow_status()
        __SLEEP_TIME = 60  # 1 minutes

        while info['status'] != "completed":
            self.set_info('Waiting until launch is done. Sleep for {} seconds.'.format(__SLEEP_TIME))
            time.sleep(__SLEEP_TIME)
            info = self.get_workflow_status()

        return info['result']
