# -*- coding: utf-8 -*-

import sandbox.common.types.task as ctt

from sandbox import common
from sandbox import sdk2
from sandbox.projects.vh.frontend.dolbilka_plan_creator import VhDolbilkaPlanCreator
from sandbox.projects.vh.frontend.stand_builder import VhStandBuilder
from sandbox.projects.vh.frontend.count_diff import VhFrontendCountDiff
from sandbox.projects.vh.frontend import (
    DolbilkaDumpResult,
    VhFrontendSqlTables,
    YABS_VH_FRONTEND_RELEASE,
)
from sandbox.projects.resource_types import (
    VH_MESSAGE_TEMPLATES,
    VH_LUA_TEMPLATES,
)


class VhBack2Back(sdk2.Task):
    """
        Run Back2Back tests for VH frontend
    """
    class Parameters(sdk2.Task.Parameters):
        plan_creator = sdk2.parameters.Task(
            "Task that created shooting plan",
            task_type=VhDolbilkaPlanCreator,
            default=None,
        )

        use_custom_tables = sdk2.parameters.Bool("Use custom tables")
        use_custom_templates = sdk2.parameters.Bool("Use custom templates")
        use_custom_lua_templates = sdk2.parameters.Bool("Use custom lua templates")

        with sdk2.parameters.Group("Pre stand") as pre_stand:
            description_pre = sdk2.parameters.String(
                "Pre stand description",
                default="Pre stand"
            )
            with use_custom_tables.value[True]:
                mysql_tables_pre = sdk2.parameters.Resource(
                    "MySQL DB for pre",
                    resource_type=VhFrontendSqlTables,
                    default=None,
                )
            with use_custom_templates.value[True]:
                template_resource_pre = sdk2.parameters.Resource(
                    "Message templates resource for pre",
                    resource_type=VH_MESSAGE_TEMPLATES,
                    default=None,
                )
            with use_custom_lua_templates.value[True]:
                template_resource_lua_pre = sdk2.parameters.Resource(
                    "Lua templates resource for pre",
                    resource_type=VH_LUA_TEMPLATES,
                    default=None,
                )
            yt_index_table_pre = sdk2.parameters.String(
                "Use custom index yt table path",
                default="//home/videoindex/full/vh/snapshots/Index"
            )

            use_prebuilt_package_pre = sdk2.parameters.Bool(
                "Use prebuilt package",
                default=False,
            )
            with use_prebuilt_package_pre.value[False]:
                arcadia_url_pre = sdk2.parameters.ArcadiaUrl(
                    "Arcadia url",
                )
            with use_prebuilt_package_pre.value[True]:
                prebuilt_package_pre = sdk2.parameters.Resource(
                    "Prebuilt package",
                    resource_type=YABS_VH_FRONTEND_RELEASE,
                )
            is_new_version_pre = sdk2.parameters.Bool(
                "New incompatible version",
                default=True
            )

        with sdk2.parameters.Group("Test stand") as test_stand:
            description_test = sdk2.parameters.String(
                "Test stand description",
                default="Test stand"
            )
            with use_custom_tables.value[True]:
                mysql_tables_test = sdk2.parameters.Resource(
                    "MySQL DB for test",
                    resource_type=VhFrontendSqlTables,
                    default=None,
                )
            with use_custom_templates.value[True]:
                template_resource_test = sdk2.parameters.Resource(
                    "Message templates resource for test",
                    resource_type=VH_MESSAGE_TEMPLATES,
                    default=None,
                )
            with use_custom_lua_templates.value[True]:
                template_resource_lua_test = sdk2.parameters.Resource(
                    "Lua templates resource for test",
                    resource_type=VH_LUA_TEMPLATES,
                    default=None,
                )
            use_prebuilt_package_test = sdk2.parameters.Bool(
                "Use prebuilt package",
                default=False,
            )
            yt_index_table_test = sdk2.parameters.String(
                "Index yt table path",
                default="//home/videoindex/full/vh/snapshots/Index"
            )
            with use_prebuilt_package_test.value[False]:
                arcadia_url_test = sdk2.parameters.ArcadiaUrl(
                    "Arcadia url",
                )
            with use_prebuilt_package_test.value[True]:
                prebuilt_package_test = sdk2.parameters.Resource(
                    "Prebuilt package",
                    resource_type=YABS_VH_FRONTEND_RELEASE,
                )
            is_new_version_test = sdk2.parameters.Bool(
                "New incompatible version",
                default=True
            )

        yt_token_name = sdk2.parameters.String(
            "YT token name in sandbox vault",
            default="yt_token",
        )

        yt_token_owner = sdk2.parameters.String(
            "YT token owner in sandbox vault",
            default=None,
        )

        yt_cluster = sdk2.parameters.String(
            "Yt cluster where yt table stored",
            default='hahn',
        )

        with sdk2.parameters.Output:
            pre_responses = sdk2.parameters.Resource(
                "Pre stand dump results",
                resource_type=DolbilkaDumpResult,
            )
            test_responses = sdk2.parameters.Resource(
                "Test stand dump results",
                resource_type=DolbilkaDumpResult,
            )

    @property
    def get_token_owner(self):
        return self.Parameters.yt_token_owner if self.Parameters.yt_token_owner is not None else self.owner

    def on_save(self):
        if self.Parameters.plan_creator is None:
            self.Parameters.plan_creator = VhDolbilkaPlanCreator.find(status=ctt.Status.Group.FINISH).first()

    def on_execute(self):
        self.create_stands()
        self.count_diff()

    def create_stands(self):
        with self.memoize_stage.create_stands:
            mysql_tables = VhFrontendSqlTables.restore(
                data=self.server.resource[{"type": "VH_FRONTEND_SQL_TABLES"}, : 1]["items"][0]
            )

            mysql_tables_pre = self.Parameters.mysql_tables_pre
            if mysql_tables_pre is None:
                mysql_tables_pre = mysql_tables
            self.Parameters.pre_responses = DolbilkaDumpResult(
                self,
                description="Pre stand dump result",
                path="pre_result.dump",
            )
            pre_stand = VhStandBuilder(
                self,
                arcadia_url=self.Parameters.arcadia_url_pre,
                description=self.Parameters.description_pre,
                dump_resource=self.Parameters.pre_responses,
                is_new_version=self.Parameters.is_new_version_pre,
                mysql_tables=mysql_tables_pre,
                plan_creator=self.Parameters.plan_creator,
                prebuilt_package=self.Parameters.prebuilt_package_pre,
                template_resource=self.Parameters.template_resource_pre,
                template_resource_lua=self.Parameters.template_resource_lua_pre,
                use_prebuilt_package=self.Parameters.use_prebuilt_package_pre,
                yt_index_table_path=self.Parameters.yt_index_table_pre,
                yt_token_name=self.Parameters.yt_token_name,
                yt_token_owner=self.get_token_owner,
                yt_cluster=self.Parameters.yt_cluster,
            ).enqueue()

            mysql_tables_test = self.Parameters.mysql_tables_test
            if mysql_tables_test is None:
                mysql_tables_test = mysql_tables
            self.Parameters.test_responses = DolbilkaDumpResult(
                self,
                description="Test stand dump result",
                path="test_result.dump",
            )
            test_stand = VhStandBuilder(
                self,
                arcadia_url=self.Parameters.arcadia_url_test,
                description=self.Parameters.description_test,
                dump_resource=self.Parameters.test_responses,
                is_new_version=self.Parameters.is_new_version_test,
                mysql_tables=mysql_tables_test,
                plan_creator=self.Parameters.plan_creator,
                prebuilt_package=self.Parameters.prebuilt_package_test,
                template_resource=self.Parameters.template_resource_test,
                template_resource_lua=self.Parameters.template_resource_lua_test,
                use_prebuilt_package=self.Parameters.use_prebuilt_package_test,
                yt_index_table_path=self.Parameters.yt_index_table_test,
                yt_token_name=self.Parameters.yt_token_name,
                yt_token_owner=self.get_token_owner,
                yt_cluster=self.Parameters.yt_cluster,
            ).enqueue()

            tasks_to_wait = [
                pre_stand,
                test_stand,
            ]

            waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
            raise sdk2.WaitTask(
                tasks_to_wait,
                waited_statuses,
                wait_all=True
            )

    def count_diff(self):
        if self.find(VhStandBuilder, status=ctt.Status.Group.SUCCEED).count != 2:
            raise common.errors.TaskFailure("One of the builds failed")
        with self.memoize_stage.count_diff:
            diff_counter = VhFrontendCountDiff(
                self,
                description="Count diff\npre: {pre}\ntest: {test}".format(
                    pre=self.Parameters.description_pre,
                    test=self.Parameters.description_test,
                ),
                pre_responses=self.Parameters.pre_responses,
                test_responses=self.Parameters.test_responses,
                plan_creator=self.Parameters.plan_creator,
            ).enqueue()

            waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
            raise sdk2.WaitTask(
                [diff_counter],
                waited_statuses,
            )
