# -*- coding: utf-8 -*-

import logging
import json

from sandbox.common import errors
from sandbox.common.types.client import Tag
from sandbox.projects.vh.frontend.generate_playlist_service_requests import PLAYLIST_SERVICE_RESPONSES, PLAYLIST_SERVICE_REQUEST_PLAN

import sandbox.projects.common.binary_task as binary_task

from sandbox import sdk2


class VhPlaylistServiceDiffResults(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Select max_request_number requests to playlist service from YT logs.
        Generate plan for servant client and initial requests.
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        client_tags = Tag.INTEL_E5_2650 & Tag.LXC & Tag.GENERIC
        execution_space = 10 * 1024
        required_ram = 2 * 1024

    class Parameters(sdk2.Task.Parameters):
        beta_responses = sdk2.parameters.Resource(
            "beta responses",
            name="beta_responses",
            resource_type=PLAYLIST_SERVICE_RESPONSES
        )

        prod_responses = sdk2.parameters.Resource(
            "prod responses",
            name="prod_responses",
            resource_type=PLAYLIST_SERVICE_RESPONSES
        )

        plan_file = sdk2.parameters.Resource(
            "plan file for servant client",
            name="plan_file",
            resource_type=PLAYLIST_SERVICE_REQUEST_PLAN
        )

        with sdk2.parameters.RadioGroup(
            "proto (for ugc) or http (usually for vh)",
            name="should_generate_requests"
        ) as requests_scheme:
            requests_scheme.values['proto'] = requests_scheme.Value('proto', default=True)
            requests_scheme.values['http'] = requests_scheme.Value('http')

        ext_params = binary_task.LastBinaryReleaseParameters()

        with sdk2.parameters.Output():
            fail_count = sdk2.parameters.Integer(
                "fail_count",
            )

    ERRORS_BETA_RESP = []
    ERRORS_PROD_RESP = []
    ERRORS_REQUEST = []
    diff_uuids = []

    def format_stream(self, output_stream):
        ysign_pos = output_stream.find("ysign1")
        comma_pos = output_stream.find(',', ysign_pos)
        output_stream = output_stream[:ysign_pos] + output_stream[comma_pos:]

        ts_pos = output_stream.find("ts=")
        slash_pos = output_stream.find("/", ts_pos)
        output_stream = output_stream[:ts_pos] + output_stream[slash_pos:]

        return output_stream

    def compare_streams(self, prod_os, beta_os):
        prod_os_corrected = self.format_stream(prod_os)
        beta_os_corrected = self.format_stream(beta_os)
        if prod_os_corrected != beta_os_corrected:
            raise errors.TaskFailure("different output streams, prod: {prod}, beta: {beta}!".format(prod=prod_os_corrected, beta=beta_os_corrected))

    def normalize_stream_type(self, stream_type):
        if stream_type == 1:
            return "ST_DASH"
        if stream_type == 2:
            return "ST_HLS"
        if stream_type == 3:
            return "ST_MSS"
        return stream_type

    def compare_stream_type(self, prod_stream_type, beta_stream_type):
        prod_normalized = self.normalize_stream_type(prod_stream_type)
        beta_normalized = self.normalize_stream_type(beta_stream_type)
        if prod_normalized != beta_normalized:
            raise errors.TaskFailure("different stream type! prod: {prod}, beta: {beta}".format(prod=prod_normalized, beta=beta_normalized))

    def compare_stream_info(self, prod_stream_infos, beta_stream_infos):
        if len(prod_stream_infos) != len(beta_stream_infos):
            raise errors.TaskFailure("different stream infos size!")
        prod_stream_infos = sorted(prod_stream_infos, key=lambda x: self.normalize_stream_type(x["StreamType"] if "StreamType" in x else ""))
        beta_stream_infos = sorted(beta_stream_infos, key=lambda x: self.normalize_stream_type(x["StreamType"] if "StreamType" in x else ""))
        for i in xrange(len(prod_stream_infos)):
            prod_stream_info = prod_stream_infos[i]
            beta_stream_info = beta_stream_infos[i]
            self.compare_stream_type(prod_stream_info["StreamType"], beta_stream_info["StreamType"])

            prod_os = prod_stream_info["OutputStream"]
            beta_os = beta_stream_info["OutputStream"]
            self.compare_streams(prod_os, beta_os)
            if "MasterPlaylist" in prod_stream_info or "MasterPlaylist" in beta_stream_info:
                self.compare_streams(prod_stream_info["MasterPlaylist"], beta_stream_info["MasterPlaylist"])

    def compare_stream_info_http_response(self, prod_stream_infos, beta_stream_infos):
        if len(prod_stream_infos) != len(beta_stream_infos):
            raise errors.TaskFailure("different streams count!")
        prod_stream_infos = sorted(prod_stream_infos, key=lambda x: self.normalize_stream_type(x["stream_type"] if "stream_type" in x else ""))
        beta_stream_infos = sorted(beta_stream_infos, key=lambda x: self.normalize_stream_type(x["stream_type"] if "stream_type" in x else ""))
        for i in xrange(len(prod_stream_infos)):
            prod_stream_info = prod_stream_infos[i]
            beta_stream_info = beta_stream_infos[i]
            self.compare_stream_type(prod_stream_info["stream_type"], beta_stream_info["stream_type"])

            prod_os = prod_stream_info["stream"]
            beta_os = beta_stream_info["stream"]
            self.compare_streams(prod_os, beta_os)
            if "master_playlist" in prod_stream_info or "master_playlist" in beta_stream_info:
                self.compare_streams(prod_stream_info["master_playlist"], beta_stream_info["master_playlist"])

    def compare_first_frames(self, prod_first_frames, beta_first_frames):
        if len(prod_first_frames) != len(beta_first_frames):
            raise errors.TaskFailure("different first frames size!")
        prod_first_frames = sorted(prod_first_frames, key=lambda x: x.get("FirstFrameUrl", ""))
        beta_first_frames = sorted(beta_first_frames, key=lambda x: x.get("FirstFrameUrl", ""))
        for i in xrange(len(prod_first_frames)):
            prod_first_frame = prod_first_frames[i]
            beta_first_frame = beta_first_frames[i]

            def _compare_values(field, prod_value, beta_value):
                if prod_value != beta_value:
                    raise errors.TaskFailure("different first frames in field: {field}, prod: {prod}, beta: {beta}!".format(field=field, prod=prod_first_frame, beta=beta_first_frame))

            _compare_values("Width", prod_first_frame.get("Width", 0), beta_first_frame.get("Width", 0))
            _compare_values("Height", prod_first_frame.get("Height", 0), beta_first_frame.get("Height", 0))
            _compare_values("TargetBitrate", prod_first_frame.get("TargetBitrate", 0), beta_first_frame.get("TargetBitrate", 0))
            _compare_values("FirstFrameUrl", prod_first_frame.get("FirstFrameUrl", ""), beta_first_frame.get("FirstFrameUrl", ""))

    def compare_single_streams(self, prod_single_stream, beta_single_stream):
        for field in ["Thumbnail", "Uuid", "PlaylistGeneration", "OutputStreamId", "TechnicalName", "DrmType", "PlaylistGeneration", "BackendId"]:
            field_in_prod = field in prod_single_stream
            field_in_beta = field in beta_single_stream
            if field_in_prod != field_in_beta:
                raise errors.TaskFailure("field {field} is differs. in prod: {prod}, in beta: {beta}".format(field=field, prod=field_in_prod, beta=field_in_beta))

            if not field_in_prod:
                continue

            if str(prod_single_stream[field]) != str(beta_single_stream[field]):
                raise errors.TaskFailure("different field: {field}, value prod: {prod}, beta: {beta}".format(field=field, prod=prod_single_stream[field], beta=beta_single_stream[field]))
        self.compare_stream_info(prod_single_stream["StreamInfo"], beta_single_stream["StreamInfo"])
#       self.compare_first_frames(prod_single_stream["FirstFrames"], beta_single_stream["FirstFrames"])

    def compare_single_streams_http_response(self, prod_single_stream, beta_single_stream):
        for field in ["technical_name", "output_stream_id", "uuid", "playlist_generation", "drm_type", "agreement", "title"]:
            field_in_prod = field in prod_single_stream
            field_in_beta = field in beta_single_stream
            if field_in_prod != field_in_beta:
                raise errors.TaskFailure("field {field} is differs. in prod: {prod}, in beta: {beta}".format(field=field, prod=field_in_prod, beta=field_in_beta))

            if not field_in_prod:
                continue

            if str(prod_single_stream[field]) != str(beta_single_stream[field]):
                raise errors.TaskFailure("different field: {field}, value prod: {prod}, beta: {beta}".format(field=field, prod=prod_single_stream[field], beta=beta_single_stream[field]))
        self.compare_stream_info_http_response(prod_single_stream["streams"], beta_single_stream["streams"])

    def compare_http_response_json(self, prod_js, beta_js, plan_str):
        prod_set = set()
        for x in prod_js:
            prod_set.add(x["uuid"])

        beta_set = set()
        for x in beta_js:
            beta_set.add(x["uuid"])

        diff_set = prod_set.difference(beta_set)
        if len(diff_set) > 0:
            for x in diff_set:
                self.diff_uuids.append(x)
            return

        prod_js = sorted(prod_js, key=lambda x: (x["uuid"], str(x["output_stream_id"]) if "output_stream_id" in x else ""))
        beta_js = sorted(beta_js, key=lambda x: (x["uuid"], str(x["output_stream_id"]) if "output_stream_id" in x else ""))

        for i in xrange(len(prod_js)):
            try:
                self.compare_single_streams_http_response(prod_js[i], beta_js[i])
            except Exception as error:
                logging.info("compare streams failed with error: {}".format(error))
                self.ERRORS_PROD_RESP.append(prod_js)
                self.ERRORS_BETA_RESP.append(beta_js)
                self.ERRORS_REQUEST.append(plan_str)
                break

    def compare_json(self, prod_js, beta_js, plan_str):
        prod_has_single_stream = "SingleStream" in prod_js
        beta_has_single_stream = "SingleStream" in beta_js

        if prod_has_single_stream != beta_has_single_stream:
            logging.info("prod_has_single_stream: %s" % prod_has_single_stream)
            logging.info("beta_has_single_stream: %s" % beta_has_single_stream)

            self.ERRORS_PROD_RESP.append(prod_js)
            self.ERRORS_BETA_RESP.append(beta_js)
            self.ERRORS_REQUEST.append(plan_str)
            return

        if not prod_has_single_stream:
            return

        prod_single_stream_array = prod_js["SingleStream"]
        beta_single_stream_array = beta_js["SingleStream"]

        if len(prod_single_stream_array) != len(beta_single_stream_array):
            self.ERRORS_PROD_RESP.append(prod_js)
            self.ERRORS_BETA_RESP.append(beta_js)
            self.ERRORS_REQUEST.append(plan_str)

            prod_set = set()
            for x in prod_single_stream_array:
                prod_set.add(x["Uuid"])

            beta_set = set()
            for x in beta_single_stream_array:
                beta_set.add(x["Uuid"])

            diff_set = prod_set.difference(beta_set)
            for x in diff_set:
                self.diff_uuids.append(x)

            return

        prod_single_stream_array = sorted(prod_single_stream_array, key=lambda x: (x["Uuid"], str(x["OutputStreamId"]) if "OutputStreamId" in x else ""))
        beta_single_stream_array = sorted(beta_single_stream_array, key=lambda x: (x["Uuid"], str(x["OutputStreamId"]) if "OutputStreamId" in x else ""))

        for i in xrange(len(prod_single_stream_array)):
            try:
                self.compare_single_streams(prod_single_stream_array[i], beta_single_stream_array[i])
            except Exception as error:
                logging.info("compare single streams failed with error: {}".format(error))
                self.ERRORS_PROD_RESP.append(prod_js)
                self.ERRORS_BETA_RESP.append(beta_js)
                self.ERRORS_REQUEST.append(plan_str)
                break

    def on_execute(self):
        scheme = self.Parameters.requests_scheme

        beta_responses_list = []
        beta_file_path = str(sdk2.ResourceData(self.Parameters.beta_responses).path)
        with open(beta_file_path, 'r') as f:
            for line in f:
                js = json.loads(line)
                beta_responses_list.append(js)

        prod_responses_list = []
        prod_file_path = str(sdk2.ResourceData(self.Parameters.prod_responses).path)
        with open(prod_file_path, 'r') as f:
            for line in f:
                js = json.loads(line)
                prod_responses_list.append(js)

        plan_list = []
        plan_file_path = str(sdk2.ResourceData(self.Parameters.plan_file).path)
        with open(plan_file_path, 'r') as f:
            for line in f:
                plan_list.append(line)

        logging.info("prod len: %s" % len(prod_responses_list))
        logging.info("beta len: %s" % len(beta_responses_list))
        if len(prod_responses_list) != len(beta_responses_list):
            raise errors.TaskFailure("different length")

        for i in xrange(len(prod_responses_list)):
            if scheme == "proto":
                self.compare_json(prod_responses_list[i], beta_responses_list[i], plan_list[i])
            elif scheme == "http":
                self.compare_http_response_json(prod_responses_list[i], beta_responses_list[i], plan_list[i])
        self.Parameters.fail_count = len(self.ERRORS_REQUEST)

        if self.Parameters.fail_count > len(prod_responses_list) * 0.1:
            raise errors.TaskFailure("Too many diff")
