# -*- coding: utf-8 -*-

import logging
import urlparse
import random
import string
from urllib import quote

from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.common import dolbilka
from sandbox.projects.vh.frontend.generate_requests_from_yt_logs import VhGenerateRequestsFromYt
from sandbox.projects.vh.frontend import (
    VH_DOLBILKA_PLAN,
    VH_DOLBILKA_REQUESTS,
)
import sandbox.common.types.task as ctt


class VhDolbilkaPlanCreator(sdk2.Task):
    """
        Create dolbilka plan for testing videohosting from day logs in YT.
        Use task VhGenerateRequestsFromYt to select requests
    """

    class Requirements(sdk2.Task.Requirements):
        environments = (PipEnvironment("yandex-yt", "0.9.17"),)
        execution_space = 10 * 1024
        required_ram = 10 * 1024

    class Parameters(sdk2.Task.Parameters):
        should_generate_requests = sdk2.parameters.Bool(
            "If True will execute VhGenerateRequestsFromYt",
            name="should_generate_requests",
            default=False,
            required=True
        )
        with should_generate_requests.value[True]:
            yt_cluster = sdk2.parameters.String(
                "YT cluster (i.e. hahn)",
                name="yt_cluster",
                default="hahn",
                required=True,
            )
            yt_token_vault = sdk2.parameters.String(
                "YT_TOKEN vault name",
                name="yt_token_vault",
                default="yt_token_for_testenv",
                required=True,
            )
            request_log_path = sdk2.parameters.String(
                "Request logs path in YT",
                name="request_log_path",
                default="//logs/frontend-vh-balancer-access-log/1d",
                required=True,
            )
            max_request_number = sdk2.parameters.String(
                "maximum requests number",
                name="max_request_number",
                default="10000"
            )
            yt_logs_limit = sdk2.parameters.String(
                "number of rows in YT table with logs to read",
                name="yt_logs_limit",
                default="1000000"
            )
            handlers_whitelist = sdk2.parameters.List(
                "If not empty only handlers from whitelist will be shot, handlers should start with /",
            )
            handlers_blacklist = sdk2.parameters.List(
                "handlers from blacklist will not be shot, handlers should start with /",
            )
            headers = sdk2.parameters.List(
                "headers for shooting",
                default="",
            )
            percentage_use_of_headers = sdk2.parameters.Integer(
                "percentage use of headers for shooting",
                name="percentage_use_of_headers",
                default="100",
            )
        with should_generate_requests.value[False]:
            raw_requests = sdk2.parameters.Resource(
                "File with requests from YT logs",
                name="raw requests",
                default=None,
                # required=True,
            )
            request_number_input = sdk2.parameters.Integer(
                "Requests number in file with requests"
            )

        host = sdk2.parameters.String(
            "host for shooting",
            name="host",
            default="localhost",
        )
        port = sdk2.parameters.String(
            "port for shooting",
            name="port",
            default="80",
        )
        shooting_delay = sdk2.parameters.String(
            "delay of shooting in microseconds",
            name="shooting_delay",
            default="1000",
        )
        additional_query_params = sdk2.parameters.List(
            "Additional query params, will be in request as &param1&param2",
            default={}
        )
        prefix = sdk2.parameters.String(
            "Prefix, will be added in every request",
            default=""
        )

        with sdk2.parameters.Output():
            request_number = sdk2.parameters.Integer(
                "requests number",
            )
            requests = sdk2.parameters.Resource(
                "used requests",
                resource_type=VH_DOLBILKA_REQUESTS,
            )
            plan = sdk2.parameters.Resource(
                "plan for shooting",
                resource_type=VH_DOLBILKA_PLAN,
            )

    def create_requests_file(self, raw_requests_path, requests_file):
        logging.info("Start update requests file")
        seed = hash(raw_requests_path)
        random.seed(seed)
        rand_str = lambda n: ''.join([random.choice(string.lowercase + string.digits) for i in xrange(n)])

        with open(str(raw_requests_path), "r") as raw_requests_file:
            with open(str(requests_file), "w") as out_file:
                for row in raw_requests_file:
                    split_row = row.strip().split('\t')
                    url = row.strip().split('\t')[0]
                    if len(split_row) > 1:
                        headers = row.strip().split('\t')[1]

                    query_params = ["shoot_reqid=" + rand_str(8)]
                    if self.Parameters.additional_query_params:
                        query_params += self.Parameters.additional_query_params
                    url = self.add_query_params(url, query_params)
                    if self.Parameters.prefix:
                        prefix = self.Parameters.prefix
                        url = prefix + url
                        if "?" in url:
                            parts = url.split("?")
                            url = parts[0] + "?" + "&".join(parts[1:])
                    plan_row = "{delay}\t{url}\t{headers}\n".format(
                        delay=self.Parameters.shooting_delay,
                        url=url.strip(),
                        headers=headers.strip(),
                    )
                    out_file.write(plan_row)

    def add_query_params(self, url, params):
        """
            Add query params to every request
        """
        parts = urlparse.urlparse(url)
        query_list = urlparse.parse_qsl(parts.query)
        for param in params:
            param_key = param.strip().split("=", 1)[0]
            param_value = param.strip().split("=", 1)[1]
            query_list.append((param_key, param_value))
        quoted_list = []
        for k, v in query_list:
            quoted_list.append((quote(k), quote(v)))
        new_parts = list(parts)
        new_parts[4] = "&".join(map(lambda kv: "{}={}".format(*kv), quoted_list))
        return urlparse.urlunparse(new_parts)

    def on_execute(self):
        raw_requests_file = None
        if self.Parameters.should_generate_requests or self.Parameters.raw_requests is None:
            with self.memoize_stage.create_children:
                requests_generator = VhGenerateRequestsFromYt(
                    self,
                    yt_cluster=self.Parameters.yt_cluster,
                    yt_token_vault=self.Parameters.yt_token_vault,
                    request_log_path=self.Parameters.request_log_path,
                    max_request_number=self.Parameters.max_request_number,
                    yt_logs_limit=self.Parameters.yt_logs_limit,
                    handlers_whitelist=self.Parameters.handlers_whitelist,
                    handlers_blacklist=self.Parameters.handlers_blacklist,
                    headers=self.Parameters.headers,
                    percentage_use_of_headers=self.Parameters.percentage_use_of_headers
                ).enqueue()
                self.Context.requests_generator_task_id = requests_generator.id
                logging.info("requests_generator.id: %s" % requests_generator.id)

                raise sdk2.WaitTask(
                    [requests_generator],
                    set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
                )

            raw_requests_file = sdk2.ResourceData(sdk2.Task[self.Context.requests_generator_task_id].Parameters.raw_requests)
            self.Parameters.request_number = sdk2.Task[self.Context.requests_generator_task_id].Parameters.request_number
        elif self.Parameters.request_number_input is not None and self.Parameters.request_number_input > 0:
            self.Parameters.request_number = self.Parameters.request_number_input
            raw_requests_file = sdk2.ResourceData(self.Parameters.raw_requests)
        else:
            raw_requests_file = sdk2.ResourceData(self.Parameters.raw_requests)
            self.Parameters.request_number = 100

        logging.info("VhGenerateRequestsFromYt finished successfully. Id %s", self.Context.requests_generator_task_id)
        raw_requests_path = raw_requests_file.path
        self.Parameters.requests = VH_DOLBILKA_REQUESTS(self, "used requests", "requests.txt")
        self.Parameters.plan = VH_DOLBILKA_PLAN(self, "plan for shooting", "dolbilka.plan")
        requests_file = sdk2.ResourceData(self.Parameters.requests).path
        plan_file = sdk2.ResourceData(self.Parameters.plan).path

        self.create_requests_file(raw_requests_path, requests_file)
        logging.info("Successfully created requests file")

        d_planner = dolbilka.DolbilkaPlanner()
        d_planner.create_plan(
            str(requests_file.resolve()),
            result_path=str(plan_file),
            loader_type="plain",
            host=self.Parameters.host,
            port=self.Parameters.port,
        )
