# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.projects import resource_types
import logging
import tarfile
from sandbox.sandboxsdk import svn
from sandbox.projects.common.utils import sync_last_stable_resource
from sandbox.projects.vh.frontend import VhAppHostBackendsGenerator, VhAppHostBackendsJson, AppHostConfigBundleVh
import os

import subprocess as sp
import shutil

from sandbox.projects.resource_types import (
    APP_HOST_GRAPH_GENERATOR_EXECUTABLE
)


class VhGenerateAppHostConfigs(sdk2.Task):
    """
        Vh app host configs generator
    """

    VIDEO_VERTICAL_ARCADIA_PATH = "arcadia:/arc/trunk/arcadia/web/app_host/conf/graph_generator/vertical"

    class Requirements(sdk2.Requirements):
        execution_space = 10 * 1024
        required_ram = 16 * 1024

    class Parameters(sdk2.Parameters):

        mode = sdk2.parameters.Bool("testing installation")

        with mode.value[True]:
            with sdk2.parameters.Group("Testing mode") as testing_mode:
                graphs_directory = sdk2.parameters.ArcadiaUrl(
                    "Path to testing graphs",
                    required=True,
                    default_value="arcadia:/arc/trunk/arcadia/extsearch/video/vh/apphost_data/testing/graphs"
                )
                inverse_backends_path_testing = sdk2.parameters.ArcadiaUrl(
                    "Path to testing VH inverse backends",
                    required=True,
                    default_value="arcadia:/arc/trunk/arcadia/extsearch/video/vh/apphost_data/testing"
                )

        with mode.value[False]:
            with sdk2.parameters.Group("Prod mode") as production_mode:
                inverse_backends_path_prod = sdk2.parameters.ArcadiaUrl(
                    "Path to production VH inverse backends",
                    required=True,
                    default_value="arcadia:/arc/trunk/arcadia/extsearch/video/vh/apphost_data/prod/"
                )

        releasing_resource = sdk2.parameters.Resource(
            "releasing app host config bundle video resource",
            resource_type=resource_types.APP_HOST_CONFIG_BUNDLE_VIDEO,
            required=True,
            default=None,
        )

    @property
    def inverse_backends_path(self):
        if self.Parameters.mode:
            return self.Parameters.inverse_backends_path_testing
        else:
            return self.Parameters.inverse_backends_path_prod

    def export_production_files(self):
        video_vertical_path = os.path.join(os.path.realpath(os.curdir), 'vertical')
        svn.Arcadia.export(self.VIDEO_VERTICAL_ARCADIA_PATH, video_vertical_path)
        logging.info("video_path: {}".format(video_vertical_path))
        return video_vertical_path

    def export_vh_files(self):
        vh_data_dir = os.path.join(os.path.realpath(os.curdir), 'vh_data')
        svn.Arcadia.export(self.inverse_backends_path, vh_data_dir)
        logging.info("testing path: {}".format(vh_data_dir))
        return vh_data_dir

    def get_vh_app_host_tool(self):
        vh_app_host_tool = sync_last_stable_resource(VhAppHostBackendsGenerator)
        return vh_app_host_tool

    def prepare_inverse_backends_file(self, vh_files_dir, prod_files_dir):
        vh_inverse_backends = os.path.join(vh_files_dir, "_inverse_backends.json")
        logging.info("vh inverse backends file: {}".format(vh_inverse_backends))
        production_inverse_backends = os.path.join(prod_files_dir, "VIDEO/_inverse_backends.json")
        logging.info("production inverse backends file: {}".format(production_inverse_backends))
        inverse_backends_generated = os.path.join(os.path.realpath(os.curdir), "_inverse_backends.generated")
        logging.info("generated inverse backends file: {}".format(inverse_backends_generated))

        vh_app_host_tool = self.get_vh_app_host_tool()
        logging.info("last stable vh app host tool synced")

        with sdk2.helpers.ProcessLog(self, logger="popen") as pl:
            pl.logger.setLevel(logging.DEBUG)
            proc = sp.Popen(
                [vh_app_host_tool, "merge_inverse", "-video", production_inverse_backends, "-vh", vh_inverse_backends, "-o", inverse_backends_generated],
                stdout=pl.stdout,
                stderr=pl.stderr
            )
            out, err = proc.communicate()
            logging.info("out: {}".format(out))
            logging.info("err: {}".format(err))

        shutil.copy2(inverse_backends_generated, os.path.join(prod_files_dir, "VIDEO/_inverse_backends.json"))
        return inverse_backends_generated

    def find_dir(self, substr, path):
        for root, dirs, files in os.walk(path):
            for name in dirs:
                if substr in name:
                    return os.path.join(root, name)

    def on_execute(self):
        vh_files_directory = self.export_vh_files()
        production_files_directory = self.export_production_files()
        inverse_backends_generated_path = self.prepare_inverse_backends_file(vh_files_directory, production_files_directory)
        graph_generator_tool = sync_last_stable_resource(APP_HOST_GRAPH_GENERATOR_EXECUTABLE)

        with sdk2.helpers.ProcessLog(self, logger="graph_generator") as pl:
            sp.Popen(
                [graph_generator_tool, "-vv", "backends", "VIDEO", "-g", production_files_directory, "-a", "app_host", "--skip-ping", "-o", production_files_directory],
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()

        vh_app_host_tool = self.get_vh_app_host_tool()

        """
        inverse_json_to_prodiction_tmp = os.path.join(production_files_directory, "_backends.VIDEO.json")
        if not os.path.isfile(inverse_json_to_prodiction_tmp):
            _backends_json = sync_last_stable_resource(VhAppHostBackendsJson)
            shutil.copy2(_backends_json, os.path.join(production_files_directory, "_backends.VIDEO.json"))
        """

        yp_env = os.environ.copy()
        yp_env["YP_TOKEN"] = sdk2.Vault.data(self.owner, "YP_OAUTH_TOKEN")

        with sdk2.helpers.ProcessLog(self, logger="update_topology") as pl:
            sp.Popen(
                [vh_app_host_tool, "apphost_backends", "-i", os.path.join(production_files_directory, "_backends.VIDEO.json"), "-o", "_backends.json"],
                env=yp_env,
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()

        resource_generated = sdk2.ResourceData(
            VhAppHostBackendsJson(
                self,
                description="VH frontend apphost generated configs",
                path=os.path.join("_backends.json")
            )
        )
        resource_generated.ready()

        video_app_host_config_bundle_path = sdk2.ResourceData(self.Parameters.releasing_resource).path
        with tarfile.open(str(video_app_host_config_bundle_path)) as video_app_host_config_bundle:
            video_app_host_config_bundle.extractall()

        video_app_host_config_bundle_dir = self.find_dir("app_host_config_bundle_VIDEO", os.curdir)
        shutil.copy2(inverse_backends_generated_path, os.path.join(video_app_host_config_bundle_dir, "_inverse_backends.json"))

        if self.Parameters.mode:
            graphs_dir = os.path.join(os.path.realpath(os.curdir), 'graphs')
            svn.Arcadia.export(self.Parameters.graphs_directory, graphs_dir)

            for root, dirs, files in os.walk(graphs_dir):
                for file_name in files:
                    if file_name != "somegraph.json":
                        shutil.copy2(os.path.join(graphs_dir, file_name), os.path.join(video_app_host_config_bundle_dir, "_ALL/graphs/{}".format(file_name)))

        result_tar_file_name = "vh_app_host_config_bundle.tar.gz"
        with tarfile.open(result_tar_file_name, "w:gz") as vh_app_host_config_bundle:
            vh_app_host_config_bundle.add(video_app_host_config_bundle_dir)

        vh_app_host_config_bundle_resource = sdk2.ResourceData(
            AppHostConfigBundleVh(
                self,
                description="VH app host config bundle",
                path=result_tar_file_name
            )
        )

        vh_app_host_config_bundle_resource.ready()
