import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.vh.frontend import (
    VhFrontendSqlTables,
    YABS_VH_FRONTEND_RELEASE,
)
from sandbox.projects.resource_types import (
    VH_MESSAGE_TEMPLATES,
    VH_LUA_TEMPLATES,
)
from sandbox.projects.vh.frontend.dolbilka_plan_creator import VhDolbilkaPlanCreator
from sandbox.projects.vh.frontend.stand_builder import VhStandBuilder
from sandbox.sandboxsdk.environments import PipEnvironment
import logging
import datetime


def median(x):
    return sorted(x)[int(len(x)/2)]


def hodges_lehmann_median(a):
    return median([(x + y)/2 for n, x in enumerate(a) for y in a[n:]])


class VhPerfResourceCreator(sdk2.Task):
    """ Task to create new pefomance tests resource snapshot for video hostin project """

    class Requirements(sdk2.Requirements):
        environments = (PipEnvironment("yandex-yt"), )

    class Parameters(sdk2.Task.Parameters):

        dry_run = sdk2.parameters.Bool("No shootings just generate initial snapshot with 1.0 coefficient")

        with dry_run.value[False]:
            prebuilt_package = sdk2.parameters.Resource(
                "Prebuilt package",
                resource_type=YABS_VH_FRONTEND_RELEASE,
                required=True,
            )

        with dry_run.value[False]:
            previous_resource_creator = sdk2.parameters.Task(
                "Task that created previous perf resource",
                task_type='VH_PERF_RESOURCE_CREATOR',
                required=True,
            )

            template_resource = sdk2.parameters.Resource(
                "Message templates resource",
                resource_type=VH_MESSAGE_TEMPLATES,
                required=True
            )

            template_resource_lua = sdk2.parameters.Resource(
                "Lua templates resource",
                resource_type=VH_LUA_TEMPLATES,
                required=True
            )

        yt_token_name = sdk2.parameters.String(
            "Yt token name",
            default='yt_token',
        )

        yt_token_owner = sdk2.parameters.String(
            "Yt token owner",
            default=None,
        )

        snapshot_table_path = sdk2.parameters.String(
            "Index table to copy",
            default='//home/videoindex/full/vh/snapshots/Index',
        )

        plan_creator = sdk2.parameters.Task(
            "Task that created shooting plan",
            task_type=VhDolbilkaPlanCreator,
            required=True,
        )

        mysql_tables = sdk2.parameters.Resource(
            "MySQL DB archive",
            resource_type=VhFrontendSqlTables,
            required=True
        )

        ttl = sdk2.parameters.Integer(
            "Days to store snapshot",
            default=127,
        )

        with sdk2.parameters.Output():
            coefficient = sdk2.parameters.Float(
                "Resources coefficient",
            )

    def on_execute(self):
        """
        Take previous resource of same type. If not exists create new one with coefficient equal to one
        If exists, take latest revision of trunk and build vh frontend.
        Take latest db snapshot and latest vh dolbilka plan
        Run perfomance tests for same binary with two resource installations (previous and current)
        Get RPS and compute coefficient for new resource snaphot from previous one
        Save json with coefficient and db, plan resource ids
        """

        self.associate_index_table()

        if self.Parameters.dry_run:
            self.Parameters.coefficient = 1.0
        else:
            with self.memoize_stage.create_children:
                self._shoot_and_compare()
            subtasks = list(self.find())
            next_stands = []
            prev_stands = []
            for subtask in subtasks:
                if self.Context.subtasks[str(subtask.id)]:
                    prev_stands.append(subtask)
                else:
                    next_stands.append(subtask)
            prev_med = hodges_lehmann_median([stand.Parameters.max_rps for stand in prev_stands])
            next_med = hodges_lehmann_median([stand.Parameters.max_rps for stand in next_stands])
            self.Parameters.coefficient = self.Parameters.previous_resource_creator.Parameters.coefficient * float(prev_med) / float(next_med)

    @property
    def yt_token_owner(self):
        if self.Parameters.yt_token_owner is None:
            return self.owner
        else:
            return self.Parameters.yt_token_owner

    @property
    def yt_token(self):
        return sdk2.Vault.data(self.yt_token_owner, self.Parameters.yt_token_name)

    def associate_index_table(self):
        self.Context.index_table = '//home/videoindex/full/vh/snapshots/Index.{}'.format(self.id)

        import yt.wrapper as yt
        yt_proxies = [
            'hahn',
            'banach',
        ]
        for yt_proxy in yt_proxies:
            yt_client = yt.YtClient(proxy="{}.yt.yandex.net".format(yt_proxy), token=self.yt_token)
            logging.info('Going to copy index snapshot table to "{0}"'.format(self.Context.index_table))
            yt_client.copy(self.Parameters.snapshot_table_path, self.Context.index_table, force=True, recursive=True)
            expiration_time = datetime.datetime.utcnow() + datetime.timedelta(days=self.Parameters.ttl)
            yt_client.set(
                self.Context.index_table + '/@expiration_time',
                expiration_time.isoformat(),
            )

    def _shoot_and_compare(self):
        next_stands = []
        for i in range(5):
            next_stand = VhStandBuilder(
                self,
                description='next perf resources {}'.format(i),
                mysql_tables=self.Parameters.mysql_tables,
                plan_creator=self.Parameters.plan_creator,
                prebuilt_package=self.Parameters.prebuilt_package,
                template_resource=self.Parameters.template_resource,
                template_resource_lua=self.Parameters.template_resource_lua,
                yt_index_table_path=self.Context.index_table,
                use_prebuilt_package=True,
                perf_run=True,
            )
            next_stand.enqueue()
            next_stands.append(next_stand)

        prev_task_params = self.Parameters.previous_resource_creator.Parameters
        prev_task_context = self.Parameters.previous_resource_creator.Context
        prev_stands = []
        for i in range(5):
            prev_stand = VhStandBuilder(
                self,
                description='previous perf resources {}'.format(i),
                mysql_tables=prev_task_params.mysql_tables,
                plan_creator=prev_task_params.plan_creator,
                prebuilt_package=self.Parameters.prebuilt_package,
                template_resource=self.Parameters.template_resource,
                template_resource_lua=self.Parameters.template_resource_lua,
                yt_index_table_path=prev_task_context.index_table,
                use_prebuilt_package=True,
                perf_run=True,
            )
            prev_stand.enqueue()
            prev_stands.append(prev_stand)

        tasks_to_wait = []
        tasks_to_wait.extend(next_stands)
        tasks_to_wait.extend(prev_stands)
        self.Context.subtasks = {}
        for task in prev_stands:
            self.Context.subtasks[str(task.id)] = True
        for task in next_stands:
            self.Context.subtasks[str(task.id)] = False

        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH))
        raise sdk2.WaitTask(
            tasks_to_wait,
            waited_statuses,
            wait_all=True
        )
