import json
import logging
import requests
from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.projects.resource_types.releasers as resource_releasers

from sandbox.projects.common.nanny import nanny


class VHRecommenderServiceCache(sdk2.Resource):
    ttl = "inf"
    releasable = True
    releasers = (resource_releasers.vh_backend_releasers.append("nullenstellat"))


class VHBinaryCacheLoader(sdk2.Resource):
    executable = True
    releasable = True


class RSCacheUploaderVTwo(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Gets cache for RecommenderServiceCache
    """

    class Context(sdk2.Task.Context):
        carousel_tags_loaded = 0
        feed_tags_loaded = 0

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('psycopg2-binary')
        ]

    class Parameters(sdk2.Parameters):
        video_hub_adress = sdk2.parameters.String(
            "Address of videohub",
            default="http://video-http-apphost.yandex.ru/video/videohub",
        )

        max_carousels = sdk2.parameters.Integer(
            "Max number of carousels can be got on one session",
            default=100,
        )

        max_tagged_carousels = sdk2.parameters.Integer(
            "Max number of carousels on special theme can be got on one session",
            default=20,
        )

        max_carousels_on_request = sdk2.parameters.Integer(
            "Max number of carousels got on one request to hub",
            default=10,
        )

        timeout = sdk2.parameters.Integer(
            "Timeout for one request(ms)",
            default=30000,
        )

        max_retry_count = sdk2.parameters.Integer(
            "Max retry number on one request",
            default=5,
        )

        connecton_string_secret = sdk2.parameters.String(
            "Name of secret with connection string",
        )

        binary_loader = sdk2.parameters.Resource(
            "Binary for making reqests",
            resource_type=VHBinaryCacheLoader,
            default=None,
        )

        crt_url = sdk2.parameters.String(
            "Url for getting sertificates",
        )

    def on_execute(self):
        carousels = self._get_carousels("carousel", self.Parameters.max_carousels, "", True)
        feed = self._get_feed()
        tags = self._get_tags()
        result_cache = {"carousels": carousels, "feed": feed, "carousel_tags": {}, "feed_tags": {}}
        for tag in tags:
            tag_carousels = self._get_carousels("carousel", self.Parameters.max_tagged_carousels, tag, False)
            if len(tag_carousels) > 0:
                result_cache["carousel_tags"][tag] = tag_carousels
                self.Context.carousel_tags_loaded += 1
            feed_tag_carousels = self._get_carousels("feed", self.Parameters.max_tagged_carousels, tag, False)
            if len(tag_carousels) > 0:
                result_cache["feed_tags"][tag] = feed_tag_carousels
                self.Context.feed_tags_loaded += 1
        cache_string = json.dumps(result_cache, indent=2)
        cache_resource = sdk2.ResourceData(
            VHRecommenderServiceCache(
                self,
                description="rs_cache_json",
                path="rs_cache.json"
            ),
        )
        cache_resource.path.write_bytes(cache_string)

    def _get_connection_params(self):
        try:
            secret_params = sdk2.Vault.data(self.Parameters.connecton_string_secret)
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Can't find connection params")
        crt_filename = "./root.crt"
        with open(crt_filename, "wb") as file:
            response = requests.get(self.Parameters.crt_url)
            file.write(response.content)
        return secret_params + " sslrootcert=" + crt_filename

    def _get_tags(self):
        tags = []
        connections_params = self._get_connection_params()
        import psycopg2
        conn = psycopg2.connect(connections_params)
        with conn.cursor() as cursor:
            cursor.execute("SELECT \"Name\" FROM ugc.\"Supertag\";")
            for row in cursor:
                tags.append(row[0])
        logging.info(str(tags))
        return tags

    def _get_raw_carousels(self, carousels_type, offset, path_to_loader, tag):
        try:
            args = [path_to_loader,
                    "--type",
                    carousels_type,
                    "-a",
                    self.Parameters.video_hub_adress,
                    "-r",
                    self.Parameters.max_retry_count,
                    "-t",
                    self.Parameters.timeout,
                    "-o",
                    offset,
                    "-l",
                    self.Parameters.max_carousels_on_request]
            if not tag == "":
                args.append("--tag")
                args.append(tag)
            args = [str(arg) for arg in args]
            logging.info(str(args))
            p = sp.Popen(args, stdout=sp.PIPE, stderr=sp.PIPE)
            std, err = p.communicate()
            logging.info(err)
            return json.loads(std)
        except Exception as e:
            raise common.errors.TaskFailure(
                "Error during request: {}".format(str(e)))

    def _get_carousel_id(self, carousel_info):
        try:
            if carousel_info["GroupType"] == "vitrina":
                return carousel_info["_SerpInfo"]["subtype"]
            else:
                return carousel_info["CategId"]
        except TypeError:
            raise common.errors.TaskFailure("Unknown format of config")
            return ""

    def _get_path_to_binary(self):
        bin_path = str(
            sdk2.ResourceData(self.Parameters.binary_loader).path.absolute())
        logging.info("path:{}".format(bin_path))
        return bin_path

    def _get_carousels(self, carousel_type, carousel_limit, tag, throw_on_error):
        result_carousels = dict()
        processed_carousels = set()
        offset = 0
        path = self._get_path_to_binary()
        carousels_on_request = self.Parameters.max_carousels_on_request
        while True:
            if offset + carousels_on_request > carousel_limit:
                break
            new_carousels = self._get_raw_carousels(carousel_type, offset, path, tag)
            offset += carousels_on_request
            if not("Groups" in new_carousels):
                if throw_on_error:
                    raise common.errors.TaskFailure(
                        "Uncorrect responce on carousels request: {}".format(str(new_carousels)))
                else:
                    return {}
            if offset == self.Parameters.max_carousels_on_request:
                result_carousels = new_carousels
                for carousel_info in new_carousels["Groups"]:
                    new_carousel_id = self._get_carousel_id(carousel_info)
                    if new_carousel_id == "":
                        break
                    processed_carousels.add(new_carousel_id)
            else:
                for carousel_info in new_carousels["Groups"]:
                    new_carousel_id = self._get_carousel_id(carousel_info)
                    if new_carousel_id == "":
                        break
                    if new_carousel_id not in processed_carousels:
                        processed_carousels.add(new_carousel_id)
                        result_carousels["Groups"].append(carousel_info)
        return result_carousels

    def _get_feed(self):
        path = self._get_path_to_binary()
        feed = self._get_raw_carousels("feed", 0, path, "")
        if not("Groups" in feed):
            raise common.errors.TaskFailure("Uncorrect responce on feed request")
        return feed
