# -*- coding: utf-8 -*-

import logging
import base64
import json
import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry

from sandbox.common import errors
from sandbox.common.types.client import Tag
from sandbox.projects.app_host.resources import AppHostServantClientExecutable
from sandbox.projects.common.utils import sync_last_stable_resource
from sandbox.projects.vh.frontend.generate_playlist_service_requests import PLAYLIST_SERVICE_REQUEST_PLAN, PLAYLIST_SERVICE_RESPONSES
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.common.binary_task as binary_task

from sandbox import sdk2


class VhShootPlaylistServiceWithServantClient(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Select max_request_number requests to playlist service from YT logs.
        Generate plan for servant client and initial requests.
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        client_tags = Tag.INTEL_E5_2650 & Tag.LXC & Tag.GENERIC
        execution_space = 10 * 1024
        required_ram = 2 * 1024

    class Parameters(sdk2.Task.Parameters):
        plan_file_vh_http = sdk2.parameters.Resource(
            "vh plan file for servant client (http requests). Choose only one: proto or http requests",
            name="raw_requests_file_vh",
            resource_type=PLAYLIST_SERVICE_REQUEST_PLAN,
            default=None
        )

        plan_file_vh = sdk2.parameters.Resource(
            "vh plan file for servant client (proto requests). Choose only one: proto or http requests",
            name="raw_requests_file_vh",
            resource_type=PLAYLIST_SERVICE_REQUEST_PLAN,
            default=None
        )

        plan_file_ugc = sdk2.parameters.Resource(
            "ugc plan file for servant client",
            name="raw_requests_file_ugc",
            resource_type=PLAYLIST_SERVICE_REQUEST_PLAN,
            default=None
        )

        host = sdk2.parameters.String(
            "host with playlist_service for shooting. Choose only one: host OR yappy beta",
            name="host_for_shooting",
            default=""
        )

        beta_name = sdk2.parameters.String(
            "generated yappy beta name. Choose only one: host OR yappy beta",
            name="generated yappy beta name",
            default=""
        )

        yappy_token_vault = sdk2.parameters.String(
            "yappy token vault name. required with yappy beta",
            name="yappy token oauth",
            default=""
        )

        ext_params = binary_task.LastBinaryReleaseParameters()

        with sdk2.parameters.Output():
            responses_vh = sdk2.parameters.Resource(
                "responses vh",
                name="binary_responses_vh",
                resource_type=PLAYLIST_SERVICE_RESPONSES
            )
            responses_ugc = sdk2.parameters.Resource(
                "responses ugc",
                name="binary_responses_ugc",
                resource_type=PLAYLIST_SERVICE_RESPONSES
            )

    def on_execute(self):
        host = self.get_host()

        if self.Parameters.plan_file_vh_http is not None:
            plan_vh_path = str(sdk2.ResourceData(self.Parameters.plan_file_vh_http).path)

            self.Parameters.responses_vh = PLAYLIST_SERVICE_RESPONSES(self, "responses from PLAYLIST_SERVICE_VH", "json_responses_vh")
            responses_vh_path = str(sdk2.ResourceData(self.Parameters.responses_vh).path)

            self.shoot_service(host + ":80/playlist", plan_vh_path, responses_vh_path, "http")
        elif self.Parameters.plan_file_vh is not None:
            plan_vh_path = str(sdk2.ResourceData(self.Parameters.plan_file_vh).path)

            self.Parameters.responses_vh = PLAYLIST_SERVICE_RESPONSES(self, "responses from PLAYLIST_SERVICE_VH", "binary_responses_vh")
            responses_vh_path = str(sdk2.ResourceData(self.Parameters.responses_vh).path)

            self.shoot_service(host + ":80/vh-streams", plan_vh_path, responses_vh_path, "proto")

        if self.Parameters.plan_file_ugc is not None:
            plan_ugc_path = str(sdk2.ResourceData(self.Parameters.plan_file_ugc).path)

            self.Parameters.responses_ugc = PLAYLIST_SERVICE_RESPONSES(self, "responses from PLAYLIST_SERVICE_UGC", "binary_responses_ugc")
            responses_ugc_path = str(sdk2.ResourceData(self.Parameters.responses_ugc).path)

            self.shoot_service(host + ":6432/ugc-streams", plan_ugc_path, responses_ugc_path, "proto")

    def get_yappy_token(self):
        return sdk2.Vault.data(self.Parameters.yappy_token_vault)

    def get_host(self):
        if not self.Parameters.host and not self.Parameters.beta_name:
            raise errors.TaskFailure("One of host or beta_name is required")

        if self.Parameters.host and self.Parameters.beta_name:
            raise errors.TaskFailure("Use only one of host or beta_name")

        if self.Parameters.host:
            return self.Parameters.host

        session = requests.Session()
        retries = Retry(total=3, backoff_factor=0.1)
        session.mount('https://yappy.z.yandex-team.ru', HTTPAdapter(max_retries=retries))
        session.headers = {
            "Content-Type": "application/json",
            "Authorization": "OAuth {}".format(self.get_yappy_token())
        }

        response = session.post(
            "https://yappy.z.yandex-team.ru/api/yappy.services.Model/retrieveBetasSlots",
            json={
                "betas_regexps": self.Parameters.beta_name,
                "allocated": True
            }
        )

        if not response.raise_for_status():
            for beta in response.json()["betaSlots"]:
                if beta["betaName"] == self.Parameters.beta_name:
                    hosts = beta["typeToSlots"]["vh-playlist-service"]["hosts"]
                    logging.info("recieved beta hosts: {}".format(hosts))
                    return hosts[0][:-3]

        raise errors.TaskFailure("No alive hosts in beta {}".format(self.Parameters.beta_name))

    def process_shooting(self, servant_client, shoot_url, plan_file, scheme):
        from extsearch.video.vh.playlist_service.library.data_structures.protos.handle_by_uuid_structs_pb2 import TStreamsByUuidResponse

        responses_array = []

        command_line = [servant_client, '--plan', plan_file, shoot_url]

        process = sp.Popen(command_line, stdout=sp.PIPE)
        out, err = process.communicate()
        logging.info("process output: ")
        for line in out.splitlines():
            js_data = {}
            try:
                js_data = json.loads(line)
            except ValueError as e:
                responses_array.append({"error" : e.message + ": " + line})
                continue
            try:
                if scheme == "proto":
                    binary = js_data["answers"][0]["binary"]
                elif scheme == "http":
                    binary = js_data["answers"][0]["content"]
            except Exception as e:
                responses_array.append({"error" : "parsing response error " + e.message + ": " + line})
                continue

            if scheme == "proto":
                resp = TStreamsByUuidResponse()
                resp.ParseFromString(base64.b64decode(binary))
            elif scheme == "http":
                resp = binary
            responses_array.append(resp)
        return responses_array

    def shoot_service(self, shoot_url, plan_file, output_file, scheme):
        from extsearch.video.vh.playlist_service.library.data_structures.protos.handle_by_uuid_structs_pb2 import TStreamsByUuidResponse
        from google.protobuf.json_format import MessageToDict

        servant_client = sync_last_stable_resource(AppHostServantClientExecutable)

        responses_array = self.process_shooting(servant_client, shoot_url, plan_file, scheme)

        with open(output_file, 'w') as f:
            for resp in responses_array:
                d = resp
                s = resp
                if scheme == "proto":
                    if isinstance(resp, TStreamsByUuidResponse):
                        d = MessageToDict(resp)
                    s = json.dumps(d)

                f.write(s)
                f.write('\n')
