# -*- coding: utf-8 -*-

from sandbox import common
import sandbox.common.types.task as ctt
import json
import logging
import sandbox.projects.YabsServerGetSQLArchive
import sandbox.projects.common.yabs.server.parameters as params
from sandbox import sdk2
import tarfile

from sandbox.common.rest import Client
from sandbox.common.proxy import NoAuth
from sandbox.projects.yabs.qa.resource_types import (
    YABS_MYSQL_RESTORE_DESCRIPTION,
    YABS_MYSQL_ARCHIVE_CONTENTS,
)
from sandbox.projects.vh.frontend import (
    MYSQL_DB_RESOURCE_PATH,
    RESTORE_DESCRIPTION,
    VhFrontendSqlTables,
)


class VhFrontendGetSqlArchive(sdk2.Task):
    """
        Task for creating VH frontend SQL database tables resource
    """
    class Parameters(sdk2.Parameters):
        archive_contents = sdk2.parameters.Resource(
            "Archive contents",
            resource_type=YABS_MYSQL_ARCHIVE_CONTENTS,
            default=None,
            required=False,
        )

    def on_execute(self):
        with self.memoize_stage.create_description_resource:
            self._create_description()

        with self.memoize_stage.create_archive:
            if self.Parameters.archive_contents is not None:
                logging.info("Using provided archive contents")
            else:
                logging.info("Creating new archive contents")
                self._create_archive_task()

                waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
                raise sdk2.WaitTask(
                    self.archive_task_id,
                    waited_statuses,
                    wait_all=True
                )

        subtasks = self.find(status=ctt.Status.Group.SUCCEED)
        if self.Parameters.archive_contents is None and subtasks.count != 1:
            raise common.errors.TaskFailure("Failed to get archive")

        archive_contents_resource = self.Parameters.archive_contents
        if archive_contents_resource is None:
            archive_task = subtasks.first()
            archive_contents_resource = YABS_MYSQL_ARCHIVE_CONTENTS.find(task=archive_task).first()

        archive_contents = sdk2.ResourceData(archive_contents_resource)

        tables_resource = sdk2.ResourceData(
            VhFrontendSqlTables(
                self,
                description="VH frontend daemon SQL tables",
                path=MYSQL_DB_RESOURCE_PATH,
            )
        )
        self.path().joinpath("mysql").mkdir()
        tables = json.loads(archive_contents.path.read_text())["tables"]
        sandbox = Client(base_url="http://sandbox.yandex-team.ru/api/v1.0", auth=NoAuth())
        for table in RESTORE_DESCRIPTION:
            resource_id = tables[table]
            common.share.skynet_get(
                skynet_id=sandbox.resource[resource_id].read()["skynet_id"],
                data_dir=str(self.path()),
                fallback_to_bb=True
            )
            with tarfile.open(str(self.path() / (table + ".tar.gz"))) as table_tar:
                table_tar.extractall(str(self.path() / "mysql"))

        with tarfile.open(str(tables_resource.path), mode="w") as tables_tar:
            tables_tar.add("mysql")

        tables_resource.ready()

    def _create_description(self):
        self.db_description = YABS_MYSQL_RESTORE_DESCRIPTION(
            self,
            description="VH frontend MySQL db restore description",
            path="rd.json"
        )

        db_description_data = sdk2.ResourceData(self.db_description)
        db_description_data.path.write_bytes(json.dumps(
            RESTORE_DESCRIPTION,
            sort_keys=True,
            indent=4,
            separators=(',', ': ')
        ))
        db_description_data.ready()

    def _create_archive_task(self):
        self.archive_task_id = self.server.task({
            "children": True,
            "context": {
                sandbox.projects.YabsServerGetSQLArchive.AdditionalRestoreDescription.name: self.db_description.id,
                sandbox.projects.YabsServerGetSQLArchive.Options.name: sandbox.projects.YabsServerGetSQLArchive.Options.NO_LM_DUMPS,
                params.BinDbList.name: "",
                params.FilterTable.name: ".*",
            },
            "description": "Get VH SQL archive",
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.cls,
            },
            "notification": [],
            "owner": self.owner,
            "type": "YABS_SERVER_GET_SQL_ARCHIVE",
        })["id"]
        self.server.batch.tasks.start.update([self.archive_task_id])
