# -*- coding: utf-8 -*-

import json
import sandbox.projects.common.build.parameters as build_params
import sandbox.common.types.task as ctt
import tarfile
import os
from sandbox.common.types.client import Tag

from sandbox import sdk2
from sandbox import common
from sandbox.projects.common import dolbilka
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params
from sandbox.projects.geobase.GeodataTreeLingStable.resource import GEODATA_TREE_LING_STABLE
from sandbox.projects.geobase.IpregLayoutStable.resource import IPREG_LAYOUT_STABLE
from sandbox.projects.vh.backend.sprite_config_uploader import ChannelSpriteConfig
from sandbox.projects.common.utils import sync_last_stable_resource, sync_resource, get_and_check_last_resource_with_attribute
from sandbox.projects.resource_types import (
    BUILD_OUTPUT,
    VH_MESSAGE_TEMPLATES,
    VH_LUA_TEMPLATES,
)
from sandbox.projects.vh.frontend import (
    DolbilkaDumpResult,
    SQL_READER_CONFIG,
    VHDatabaseSnapshot,
    VhFrontendSqlTables,
    YABS_VH_FRONTEND_RELEASE,
    VH_FAST_SETTINGS,
)
from sandbox.projects.vh.frontend.stand_builder.daemon import Daemon
from sandbox.projects.vh.frontend.dolbilka_plan_creator import VhDolbilkaPlanCreator
from sandbox.projects.antiadblock.aab_all_cookies import AntiadblockAllCookies
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.path import Path

ARCADIA_BINARIES_PATH = Path("yabs/vh/frontend/test/back2back")
ARCHIVE_BINARIES_PATH = Path("video-hosting-package/linux")
ARCHIVE_BINARIES_PATH_MKDB = Path("abs/vh/frontend/package/linux")

SQL_BASES_TYPES = [
    "tmpl",
    "const"
]

YT_BASES_TYPES = [
    "cinf",
]


class VhStandBuilder(sdk2.Task):
    """
        Set up VH frontend stand for testing
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        client_tags = Tag.INTEL_E5_2650 & Tag.LXC & Tag.GENERIC
        execution_space = 10 * 1024
        required_ram = 16 * 1024

    class Parameters(sdk2.Parameters):
        plan_creator = sdk2.parameters.Task(
            "Task that created shooting plan",
            task_type=VhDolbilkaPlanCreator,
            required=True,
        )
        dump_resource = sdk2.parameters.ParentResource(
            "Resource to post dump result to",
            # type=DolbilkaDumpResult,  # FIXME: invalid argument (SANDBOX-6404)
            default=None,
        )

        perf_run = sdk2.parameters.Bool(
            "Run in perfomance mode and get maximum rps",
            default=False,
        )

        use_custom_host = sdk2.parameters.Bool(
            "Use custom host",
            default=False,
        )

        yt_index_table_path = sdk2.parameters.String(
            "YT path to index table",
            default="//home/videoindex/full/vh/prevdata/Index",
        )

        yt_token_name = sdk2.parameters.String(
            "YT token name in sandbox vault",
            default=None,
        )

        yt_token_owner = sdk2.parameters.String(
            "Yt token owner in sandbox vault",
            default=None,
        )

        yt_cluster = sdk2.parameters.String(
            "Yt cluster (hahn/banach)",
            default="hahn",
        )

        yt_delta_path = sdk2.parameters.String(
            "Yt path with fresh Content{Group,Resource}",
            default="//home/video-hosting/fresh/",
        )

        yt_dynamic_path = sdk2.parameters.String(
            "Yt path for dynamic tables",
            default="//home/video-hosting/index/dynamic/Index",
        )

        yt_traits_path = sdk2.parameters.String(
            "Yt path with Content{Category,Template}",
            default="//home/video-hosting/base/",
        )

        with use_custom_host.value[True]:
            custom_host = sdk2.parameters.String(
                "Custom host to shoot at",
                default="localhost",
            )
            custom_port = sdk2.parameters.Integer(
                "Custom port",
                default=80,
            )

        with use_custom_host.value[False]:
            use_prebuilt_package = sdk2.parameters.Bool(
                "Use prebuilt package",
                default=False,
            )
            with use_prebuilt_package.value[False]:
                arcadia_url = sdk2.parameters.ArcadiaUrl(
                    "Arcadia url",
                    required=True,
                )
            with use_prebuilt_package.value[True]:
                prebuilt_package = sdk2.parameters.Resource(
                    "Prebuilt package",
                    resource_type=YABS_VH_FRONTEND_RELEASE,
                )
            template_resource = sdk2.parameters.Resource(
                "Message templates resource",
                resource_type=VH_MESSAGE_TEMPLATES,
                default=None,
            )
            template_resource_lua = sdk2.parameters.Resource(
                "Lua templates resource",
                resource_type=VH_LUA_TEMPLATES,
                default=None,
            )
            mysql_tables = sdk2.parameters.Resource(
                "MySQL DB archive",
                resource_type=VhFrontendSqlTables,
                required=True
            )

            is_new_version = sdk2.parameters.Bool(
                "New incompatible version",
                default=True
            )

        with sdk2.parameters.Output():
            max_rps = sdk2.parameters.Integer(
                "Maximum rps from shooting sessions"
            )

    @property
    def get_yt_token(self):
        owner = self.Parameters.yt_token_owner if self.Parameters.yt_token_owner is not None else self.owner
        yt_token = self.Parameters.yt_token_name if self.Parameters.yt_token_name is not None else "yt_token"
        return sdk2.Vault.data(owner, yt_token)

    def on_execute(self):
        if self.Parameters.use_custom_host:
            self._start_shoot(
                daemon_port=self.Parameters.custom_port,
                host=self.Parameters.custom_host,
            )
        else:
            if not self.Parameters.use_prebuilt_package:
                with self.memoize_stage.build_targets:
                    self._ya_make()

                subtasks = self.find(status=ctt.Status.Group.SUCCEED)
                if not subtasks.count:
                    raise common.errors.TaskFailure("ya make failed")
                binaries = self._get_bin_path(subtasks.first())
            else:
                binaries = self._get_prebuilt_bin_path()

            bases = self._prepare_bases(binaries)
            if self.Parameters.perf_run:
                self._run_perfs(binaries, bases)
            else:
                self._run_tests(binaries, bases)

    def _apt_update(self):
        with sdk2.helpers.ProcessLog(self, "apt_get_update") as pl:
            sp.check_call(
                ["sudo", "apt-get", "update"],
                stdout=pl.stdout,
                stderr=pl.stdout
            )

    def _wake_up_mysql(self):
        with sdk2.helpers.ProcessLog(self, "mysql_install") as pl:
            self._apt_update()

            packages = [
                "mysql-client-5.5",
                "mysql-server-5.5",
            ]
            sp.check_call(
                ["sudo", "apt-get", "-y", "--force-yes", "install"] + packages,
                stdout=pl.stdout,
                stderr=pl.stdout
            )

        with sdk2.helpers.ProcessLog(self, "mysql_set_up") as pl:
            MYSQL_CREATE_DBS = \
                "create database if not exists vh; " \
                "create database if not exists yabsdb;"

            sp.check_call(
                ["mysql", "-v", "-u", "root", "-e", MYSQL_CREATE_DBS],
                stdout=pl.stdout,
                stderr=pl.stdout,
            )

            mysql_tables = sdk2.ResourceData(self.Parameters.mysql_tables)
            with tarfile.open(str(mysql_tables.path)) as tables_tar:
                tables_tar.extractall("/var/lib")

            sp.check_call(
                ["chown", "-R", "mysql:mysql", "/var/lib/mysql"]
            )

            sp.check_call(
                ["mysql", "-v", "-u", "root", "vh", "-e", "show tables;"],
                stdout=pl.stdout,
                stderr=pl.stdout,
            )
            sp.check_call(
                ["mysql", "-v", "-u", "root", "yabsdb", "-e", "show tables;"],
                stdout=pl.stdout,
                stderr=pl.stdout,
            )

    def _sql_reader_config(self):
        SQL_READER_CONFIG["password"] = sdk2.Vault.data("vh_mkdb_password")
        SQL_READER_CONFIG["sandbox_token"] = sdk2.Vault.data("vh_mkdb_sandbox_token")
        if self.Parameters.mysql_tables is not None:
            SQL_READER_CONFIG["hosts"] = ["localhost"]
            SQL_READER_CONFIG["user"] = "root"
            SQL_READER_CONFIG["password"] = ""

        if self.Parameters.template_resource is not None:
            SQL_READER_CONFIG["template_resource_id"] = self.Parameters.template_resource.id
        if self.Parameters.template_resource_lua is not None:
            SQL_READER_CONFIG["lua_template_resource_id"] = self.Parameters.template_resource_lua.id

        return json.dumps(SQL_READER_CONFIG)

    def _prepare_bases(self, binaries):
        if self.Parameters.mysql_tables is not None:
            self._wake_up_mysql()
        bases = sdk2.ResourceData(
            VHDatabaseSnapshot(
                self,
                description="DB snapshot",
                path="bases",
            ),
        )
        bases.path.mkdir(parents=True)
        self._generate_bases(
            binaries=binaries,
            bases=bases.path,
        )
        bases.ready()
        return bases.path

    def _generate_bases(self, binaries, bases):
        sql_reader_config = self._sql_reader_config()
        for sql_base_type in SQL_BASES_TYPES:
            with sdk2.helpers.ProcessLog(self, "sql_reader_" + sql_base_type) as pl:
                sql_reader = sp.Popen(
                    [
                        str(binaries / "sql_reader"),
                        "--sandbox-run",
                        "--type", sql_base_type
                    ],
                    stdin=sp.PIPE,
                    stdout=sp.PIPE,
                    stderr=pl.stdout,
                )
                raw_data = sql_reader.communicate(sql_reader_config)[0].strip()
            with sdk2.helpers.ProcessLog(self, "mkdb_" + sql_base_type) as pl:
                mkdb = sp.Popen(
                    [str(binaries / "mkdb"), str(bases / ("flat." + sql_base_type)), sql_base_type],
                    stdin=sp.PIPE,
                    stderr=pl.stdout,
                )
                mkdb.communicate(raw_data)
        geobase_path = sync_last_stable_resource(GEODATA_TREE_LING_STABLE)
        self.__geobase_path = os.path.join(os.getcwd(), geobase_path)
        ipreg_layout_file_path = sync_last_stable_resource(IPREG_LAYOUT_STABLE)
        sprite_config_file_path = sync_last_stable_resource(ChannelSpriteConfig)
        self.__ipreg_layout_file_path = os.path.join(os.getcwd(), ipreg_layout_file_path)
        self.__sprite_config_file_path = os.path.join(os.getcwd(), sprite_config_file_path)
        fast_settings_path = sync_last_stable_resource(VH_FAST_SETTINGS)
        self.__fast_settings_path = os.path.join(os.getcwd(), fast_settings_path)
        abd_config_id = get_and_check_last_resource_with_attribute(AntiadblockAllCookies).id
        abd_config_path = sync_resource(abd_config_id)
        self.__abd_config_path = os.path.join(os.getcwd(), abd_config_path)
        for yt_base_type in YT_BASES_TYPES:
            with sdk2.helpers.ProcessLog(self, "dyndb_" + yt_base_type) as pl:
                sp.Popen(
                    [
                        str(binaries / "dyndb"),
                        "--server", self.Parameters.yt_cluster,
                        "--static", self.Parameters.yt_index_table_path,
                        "--dynamic", self.Parameters.yt_dynamic_path,
                        "--delta-path", self.Parameters.yt_delta_path,
                        "--traits-path", self.Parameters.yt_traits_path,
                        "--flat", str(bases / ("flat." + yt_base_type)),
                        "--geobase", self.__geobase_path,
                        "--version", "unknown",
                    ],
                    stdin=sp.PIPE,
                    stdout=pl.stderr,
                    stderr=pl.stdout,
                    env={"YT_TOKEN": self.get_yt_token},
                ).wait()

    def _run_tests(self, binaries, bases):
        with sdk2.helpers.ProcessLog(self, "vh_daemon") as pl:
            with Daemon(
                self,
                binary_path=str(binaries / "daemons"),
                bases_path=str(bases),
                fast_config_path=self.__fast_settings_path,
                geodata_file_path=self.__geobase_path,
                ipreg_layout_file_path=self.__ipreg_layout_file_path,
                sprite_config_file_path=self.__sprite_config_file_path,
                abd_config_path=self.__abd_config_path,
                stdout=pl.stdout,
                logger=pl.logger
            ) as daemon_port:
                self._start_shoot(daemon_port)

    def _run_perfs(self, binaries, bases):
        with sdk2.helpers.ProcessLog(self, "vh_daemon") as pl:
            daemon = Daemon(
                self,
                binary_path=str(binaries / "daemons"),
                bases_path=str(bases),
                fast_config_path=self.__fast_settings_path,
                geodata_file_path=self.__geobase_path,
                ipreg_layout_file_path=self.__ipreg_layout_file_path,
                sprite_config_file_path=self.__sprite_config_file_path,
                abd_config_path=self.__abd_config_path,
                stdout=pl.stdout,
                logger=pl.logger
            )
            plan_creator = self.Parameters.plan_creator
            setattr(self.Context, dolbilka.DolbilkaExecutorRequestsLimit.name, plan_creator.Parameters.request_number)
            setattr(self.Context, dolbilka.DolbilkaSessionsCount.name, 10)
            setattr(self.Context, dolbilka.DolbilkaExecutorTimeLimit.name, 1200)

            d_executor = dolbilka.DolbilkaExecutor()
            d_executor.sessions = 10

            plan = sdk2.ResourceData(plan_creator.Parameters.plan)

            results = d_executor.run_sessions(str(plan.path), daemon, host='localhost', run_once=True)

            ctx = {}  # create it for sand balalaika
            dolbilka.DolbilkaPlanner.fill_rps_ctx(results, ctx)
            self.Parameters.max_rps = ctx['max_rps']
            pl.logger.info("Dolbilka: max_fail_rate: {}".format(ctx['max_fail_rate']))
            pl.logger.info("Dolbilka: min_notfound_rate: {}".format(ctx['min_notfound_rate']))

    def _start_shoot(self, daemon_port, host="localhost"):
        plan_creator = self.Parameters.plan_creator
        request_number = plan_creator.Parameters.request_number
        plan = sdk2.ResourceData(plan_creator.Parameters.plan)

        setattr(self.Context, dolbilka.DolbilkaOutputLenval32.name, True)
        setattr(self.Context, dolbilka.DolbilkaMaximumSimultaneousRequests.name, 1)
        setattr(self.Context, dolbilka.DolbilkaExecutorRequestsLimit.name, request_number)

        if self.Parameters.dump_resource is None:
            dolbilka_dump_result = DolbilkaDumpResult(self, "dolbilka dump", "result.dump")
        else:
            dolbilka_dump_result = self.Parameters.dump_resource
        d_executor = dolbilka.DolbilkaExecutor()
        d_executor.run_session(
            str(plan.path),
            dump=str(dolbilka_dump_result.path),
            save_answers=True,
            host=host,
            port=daemon_port,
        )

    def _get_prebuilt_bin_path(self):
        bin_path = self.path("bin")

        archive_path = sdk2.ResourceData(self.Parameters.prebuilt_package).path
        with tarfile.open(str(archive_path)) as binaries_tar:
            binaries_tar.extractall()
            with tarfile.open(str(ARCHIVE_BINARIES_PATH / "vh.tgz")) as common_binaries:
                common_binaries.extractall(str(bin_path))
            with tarfile.open(str(ARCHIVE_BINARIES_PATH / "sql_reader.tgz")) as common_binaries:
                common_binaries.extractall(str(bin_path))
            with tarfile.open(str(ARCHIVE_BINARIES_PATH / "ytdb.tgz")) as ytdb_binaries:
                ytdb_binaries.extractall(str(bin_path))
            with tarfile.open(str(ARCHIVE_BINARIES_PATH / "dyndb.tgz")) as dyndb_binaries:
                dyndb_binaries.extractall(str(bin_path))
            if self.Parameters.is_new_version:
                with tarfile.open(str(ARCHIVE_BINARIES_PATH / "mkdb.tgz")) as mkdb_binaries:
                    mkdb_binaries.extractall(str(bin_path))
            else:
                with tarfile.open(str(ARCHIVE_BINARIES_PATH_MKDB / "mkdb.tgz")) as mkdb_binaries:
                    mkdb_binaries.extractall(str(bin_path))
        return bin_path

    def _ya_make(self):
        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
        raise sdk2.WaitTask(
            self._create_ya_make_subtask(),
            waited_statuses,
            wait_all=True
        )

    def _get_bin_path(self, ya_make_task):
        build_output_queue = BUILD_OUTPUT.find(task=ya_make_task)
        if not build_output_queue.count:
            raise common.errors.TaskFailure("Can't find built targets. Failing")
        build_output = build_output_queue.first()
        return sdk2.ResourceData(build_output).path / ARCADIA_BINARIES_PATH

    def _create_ya_make_subtask(self):
        task_id = self.server.task({
            "children": True,
            "context": {
                build_params.ArcadiaUrl.name: self.Parameters.arcadia_url,
                build_params.BuildSystem.name: "ya",
                build_params.CheckReturnCode.name: True,
                build_params.CheckoutParameter.name: True,
                build_params.ClearBuild.name: False,
                build_params.TestParameter.name: True,
                get_arcadia_project_base_target_params().TargetsParameter.name: str(ARCADIA_BINARIES_PATH),
            },
            "description": "Building VH server",
            "notification": [],
            "owner": self.owner,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.cls,
            },
            "type": "YA_MAKE",
        })["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id
