# -*- coding: utf-8 -*-

import json
import requests
import time
import os

from sandbox import common
from sandbox.projects.common.search.components import SearchComponent
from sandbox.projects.common import error_handlers as eh
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import svn


DAEMON_CONFIG = {
    "BasesDirectory": "",
    "ServicePort": "",
    "ServerMaxConnectionsCount": 1000,
    "ServerQueueThreadCount": 1,
    "ServerQueueSize": 100,
    "TaskProcessingQueueThreadCount": 16,
    "TaskProcessingQueueSize": 100,
    "ExternalRequestQueueThreadCount": 4,
    "ExternalRequestQueueSize": 100,
    "ExternalRequestAttemptsCount": 3,
    "ExternalRequestTimeoutInMilliseconds": 1000,
    "ServiceThreadCount": 4,
    "MaxQueueSize": 25,
    "PublicHandlePrefix": "/public",
    "HandlerSettings": {
        "ChannelsInfoPath": "/public/channels",
        "ContentPlayerPath": "/public/player",
        "EpisodesByChannel": "/public/episodes",
        "EpisodesByProgram": "/public/program_episodes",
        "IndexPath": "/public/",
        "PingPath": "/ping",
        "RecommendedEpisodesByGenre": "/public/genre_episodes",
        "ShutdownPath": "/stop",
        "StatisticsPath": "/statistics",
        "VodEpisodesPath": "/public/vod_episodes",
        "VodLibrariesPath": "/public/vod_libraries",
        "BlackWindowsPath": "/public/black_windows",
        "GolovalStatisticsPath": "/golovan/statistics",
        "RecommendationsPath": "/public/carousels",
        "RegionsByStrmNamePath": "/strm_regions",
        "ChannelsRegions": "/channels_regions",
        "Doc2DocRecommendationsPath": "/public/doc2doc",
        "CarouselRecommendationsById": "/public/carousel",
        "License": "/public/license",
        "RecommendationsVideoHubPath": "/public/carousels_videohub",
        "CarouselRecommendationsByIdVideoHub": "/public/carousel_videohub",
        "EmbedPath": "/embed",
        "ChannelsRegionsPrivate": "/channels_regions",
        "Docs": "/docs",
        "SyncCarouselsPath": "/sync_carousels",
        "FeedPath": "/feed",
        "DrmKeysPath": "/drm_keys",
        "SeriesSeasonsPath": "/public/series_seasons",
        "SeriesEpisodesPath": "/public/series_episodes",
        "SearchPath": "/search"
    },
    "LicenseServiceSettings": {
        "BaseServiceAddress": "",
        "ExperimentServiceAddress": "http://ls.ott.yandex.net/",
        "ExperimentAllowedServices": ["ya-main", "ya-video", "ya-serp", "ya-station"],
        "ExperimentDefaultService": "ya-main",
        "ExperimentServicePath": "/v1/search/content-tree?",
        "RetryCount": 1,
        "Timeout": 50
    },
    "RecommenderServiceSettings": {
        "BaseServiceAddress": "",
        "Doc2DocRecommendationsAddress": "https://yandex.ru/video/result?",
        "NewFormatServiceAddress": "http://video-http-apphost.yandex.ru/video/videohub?",
        "PlayerTemplateUrl": "https://frontend.vh.yandex.ru/player/",
        "RetryCount": 1,
        "Timeout": 50,
        "FailOnStartIfError": False
    },
    "PlayerSandboxUrl": "https://yastatic.net/video-player/",
    "PlayerSandboxVersion": "0xac8eac6144",
    "FastSettingsRefreshTimeout": 10,
    "CaheSizeInElements": 5 * 1024,
    "CacheInvalidationIntervalInMinutes": 3,
    "LockGeoBase": False,
    "YdbEndpoint": "ydb-ru-vla-1018.search.yandex.net:31011",
    "YdbDatabase": "/ru/home/smalashkevich/mydb",
    "YdbMaxRetries": 10,
    "YdbSessionTimeoutInMs": 200
}


class Daemon(SearchComponent):
    HOST = "localhost"
    PORT = 15453
    LICENSE_PORT = 15454

    PING_COUNT = 240
    PING_INTERVAL = 2

    CONFIG_PATH = "./daemon_config"
    PROTECTED_CHANNELS_PATH = "./protected_channels.json"

    def __init__(self, task, binary_path, bases_path, fast_config_path, stdout, logger, geodata_file_path, ipreg_layout_file_path, sprite_config_file_path, abd_config_path, run_profile=False):
        super(Daemon, self).__init__(task)

        self.__logger = logger
        self.__binary_path = binary_path
        self.__fast_config_path = fast_config_path
        self.__stdout = stdout
        self.__run_profile = run_profile
        self._write_configs(bases_path, fast_config_path, geodata_file_path, ipreg_layout_file_path, sprite_config_file_path, abd_config_path)
        self.port = self.PORT

    def __enter__(self):
        self.start()
        self.wait()
        return self.PORT

    def __exit__(self, exception_type, exception_val, exception_traceback):
        self.stop()

    def _write_configs(self, bases_path, fast_config_path, geodata_file_path, ipreg_layout_file_path, sprite_config_file_path, abd_config_path):
        DAEMON_CONFIG["BasesDirectory"] = bases_path
        DAEMON_CONFIG["ServicePort"] = self.PORT
        DAEMON_CONFIG["LicenseServiceSettings"]["BaseServiceAddress"] = "http://localhost:{port}".format(port=self.LICENSE_PORT)
        DAEMON_CONFIG["GeoDataFileName"] = geodata_file_path
        DAEMON_CONFIG["SubnetsDataFileName"] = ipreg_layout_file_path
        DAEMON_CONFIG["SpriteConfig"] = sprite_config_file_path
        DAEMON_CONFIG["AbdConfig"] = abd_config_path
        self.__logger.info("Daemon config: {0}".format(DAEMON_CONFIG))
        with open(self.CONFIG_PATH, "w") as config:
            config.write(json.dumps(DAEMON_CONFIG))
        with open(self.PROTECTED_CHANNELS_PATH, "w") as protected_channels:
            protected_channels.write(json.dumps({}))

    def get_request_daemon(self, handler_name):
        return requests.get(
            "http://{host}:{port}{path}".format(
                host=self.HOST,
                port=self.PORT,
                path=DAEMON_CONFIG["HandlerSettings"][handler_name]
            )
        )

    def ping_daemon(self):
        return self.get_request_daemon("PingPath")

    def kill_daemon(self):
        return self.get_request_daemon("ShutdownPath")

    def __gen_start_options(self):
        start_opt = [] if not self.__run_profile else [self.__ya_path, 'profile', '--format=svg']
        start_opt.extend([self.__binary_path, self.CONFIG_PATH, self.__fast_config_path, self.PROTECTED_CHANNELS_PATH])
        return start_opt

    def __prepare(self):
        if self.__run_profile:
            self.__ya_path = svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', os.path.realpath('ya'))

    def start(self):
        self.__prepare()
        self.__daemon = sp.Popen(
            self.__gen_start_options(),
            stdout=self.__stdout,
            stderr=self.__stdout,
            env={'STRM_SECRET': '12345', 'YDB_TOKEN': '12345', 'DRM_AUTH_KEY': '1234567890', 'DRM_KEK': 'MTIzNDU2Nzg5MAo='}
        )
        self.__logger.info("Daemon forked")

    def wait(self):
        for i in range(self.PING_COUNT):
            try:
                response_code = self.ping_daemon().status_code
                if response_code == requests.codes.ok:
                    self.__logger.info("Daemon started")
                    return
                else:
                    self.__logger.error("Daemon responded with {code} code".format(code=response_code))
            except requests.exceptions.ConnectionError:
                self.__logger.error("Daemon is not started yet")
                if self.__daemon.poll() is not None:
                    raise common.errors.TaskFailure("Daemon died")
            time.sleep(self.PING_INTERVAL)
        raise common.errors.TaskFailure("Daemon not started")

    def stop(self):
        self.kill_daemon()
        if self.__daemon.wait():
            self.__logger.error("Daemon didn't manage to stop gracefully")

    def use_component(self, work_func):
        try:
            return work_func()
        except Exception:
            self.__logger.info("Exception at search::components::use_component:\n%s", eh.shifted_traceback())
            self._wait_coredump()
            raise
        finally:
            if self.is_running():
                self.__logger.info("Component is running")
            else:
                self.__logger.info("Component is not running")
                # self._process_post_mortem()
