import json

from sandbox import sdk2
from sandbox.common import projects_handler
from sandbox.common import utils
from sandbox.common.types import task as ctt
from sandbox.projects.common import error_handlers as eh


class TaskTypeParameter(sdk2.parameters.String):
    @utils.classproperty
    def choices(self):
        return sorted((t_type, t_type) for t_type in projects_handler.load_project_types(reuse=True))


class ReleaseTaskWithParams(sdk2.Task):
    """
        Release of task with given resources and string params
    """
    class Parameters(sdk2.Task.Parameters):
        task_type = TaskTypeParameter("Task Type", required=True)

        string_params = sdk2.parameters.String(
            "Json with string params for task ",
            default="",
        )

        resources_params = sdk2.parameters.String(
            "Json with resources params for task (name - id)",
            default="",
        )

    def on_execute(self):
        task_type = self.Parameters.task_type
        kwargs_dict = json.loads(self.Parameters.string_params)
        res_dict = json.loads(self.Parameters.resources_params)
        for name, res_id in res_dict.iteritems():
            resource = sdk2.Resource.find(id=res_id).limit(1).first()
            eh.verify(resource, "resource with id {} not found".format(res_id))
            kwargs_dict[name] = resource

        with self.memoize_stage.launch_subtask:
            self.set_info('Create build task with params {params}'.format(
                params=str(kwargs_dict)
                ))
            subtask = sdk2.Task[task_type]
            launch = subtask(
                self,
                owner=self.owner,
                **kwargs_dict
            ).enqueue()

            raise sdk2.WaitTask(
                [launch],
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )
        build_task = self.find(children=True).first()
        if build_task.status != ctt.Status.SUCCESS:
            eh.fail('Build dcron scripts, ID: {task_id} was failed.'.format(task_id=build_task.id))
        else:
            self.set_info('Create release...')

            self.server.release(
                task_id=build_task.id,
                type=ctt.ReleaseStatus.STABLE,
                subject="Release"
            )
