from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.utils import sync_last_stable_resource
from sandbox.projects.awacs.BuildAwacsCtlBinary import AWACS_CTL_LINUX
from upstream_builder import UpstreamBuilder
from sandbox import sdk2
import logging
import time


class VhCreateBalancerVersion(sdk2.Task):
    """ Task to generate lua templates from trunk """

    class Requirements(sdk2.Task.Requirements):
        arch = 'linux'

    class Parameters(sdk2.Task.Parameters):

        is_testing = sdk2.parameters.Bool(
            "Deploy version on testing component",
            default=True,
        )

        with is_testing.value['False']:
            location = sdk2.parameters.String(
                "Location to create upstream (man/sas/vla)",
                required=True,
            )

        version = sdk2.parameters.String(
            "Set upstream version (e.g. v1)",
            required=True,
        )

        backends = sdk2.parameters.String(
            "Backends numbers, divided by comma",
            required=True,
        )

    prod_namespace_template = '{location}.frontend.vh.yandex.ru'
    testing_namespace = 'vh-testing.yandex.net'

    command_template = r"{{awacsctl_binary}} --verbose --config awacs_url https://awacs.yandex-team.ru/ --config awacs_token {{awacsctl_token}} {command} {{namespace_id}}"

    @property
    def get_new_upstream_file_name(self):
        return '{}/upstreams/https_{}.yml'.format(self.get_awacs_namespace, self.Parameters.version)

    @property
    def get_push_command(self):
        return self.command_template.format(command='push')

    @property
    def get_clone_command(self):
        return self.command_template.format(command='clone')

    @property
    def get_awacs_namespace(self):
        return self.testing_namespace if self.Parameters.is_testing else self.prod_namespace_template.format(self.Parameters.location)

    @property
    def get_awacsctl_token(self):
        return sdk2.Vault.data(self.owner, 'NANNY-AWACSCTL-TOKEN')

    @property
    def awacsctl_binary(self):
        return sync_last_stable_resource(AWACS_CTL_LINUX)

    def form_command(self, command, awacsctl_binary_path, token, namespace):
        logging.info(command.format(awacsctl_binary=awacsctl_binary_path, awacsctl_token=token, namespace_id=namespace))
        return command.format(awacsctl_binary=awacsctl_binary_path, awacsctl_token=token, namespace_id=namespace).split()

    def clone_namespace(self):
        with sdk2.helpers.ProcessLog(self, logger='awacs_output') as pl:
            sp.Popen(
                self.form_command(self.get_clone_command, self.awacsctl_binary, self.get_awacsctl_token, self.get_awacs_namespace),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()

    def push_namespace(self):
        with sdk2.helpers.ProcessLog(self, logger='awacs_output') as pl:
            sp.Popen(
                self.form_command(self.get_push_command, self.awacsctl_binary, self.get_awacsctl_token, self.get_awacs_namespace),
                stdout=pl.stdout,
                stderr=pl.stderr
            ).wait()

    def parse_backends_array(self):
        backends_list = self.Parameters.backends.replace(' ', '').split(',')
        return list(filter(lambda backend_num: backend_num.isdigit(), backends_list))

    def create_upstream(self):
        logging.info(self.parse_backends_array())
        upstream_builder = UpstreamBuilder(self.Parameters.version, self.parse_backends_array())
        logging.info(upstream_builder.generate_config())
        with open(self.get_new_upstream_file_name, 'w') as new_upstream:
            new_upstream.write(upstream_builder.generate_config())

        with sdk2.helpers.ProcessLog(self, logger='ls_output') as pl:
            sp.Popen(['ls', 'vh-testing.yandex.net'], stdout=pl.stdout, stderr=pl.stderr).wait()
            sp.Popen(['ls', 'vh-testing.yandex.net/upstreams'], stdout=pl.stdout, stderr=pl.stderr).wait()

    def on_execute(self):
        self.clone_namespace()
        self.create_upstream()
        time.sleep(600)
        self.push_namespace()
