# -*- coding: utf-8 -*-

import logging
import json

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common import dolbilka
from sandbox.projects.vh.frontend import DolbilkaDumpResult, VhDolbilkaShootingStats
from sandbox.projects.vh.frontend.dolbilka_plan_creator import VhDolbilkaPlanCreator
import sandbox.projects.resource_types.releasers as resource_releasers
import sandbox.common.types.task as ctt
from sandbox.projects.vh.frontend import (
    VhDolbilkaRawRequests,
)


class DolbilkaShootingSuccess(sdk2.Resource):
    """
        Result of VhDolbilkaShooter (bool)
    """
    share = True
    releasers = (resource_releasers.vh_backend_releasers.append("fawnd2"))


class VhDolbilkaShooter(sdk2.Task):
    """
        Shooting host
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        client_tags = Tag.INTEL_E5_2650 & Tag.LXC & Tag.GENERIC
        execution_space = 10 * 1024
        required_ram = 16 * 1024

    class Parameters(sdk2.Parameters):
        plan_creator = sdk2.parameters.Task(
            "Task that created shooting plan."
            "If None will execute VhDolbilkaPlanCreator with default params",
            # task_type=VhDolbilkaPlanCreator,
            default=None,
        )
        custom_host = sdk2.parameters.String(
            "Host to shoot at",
            default="localhost",
            required=True
        )
        custom_port = sdk2.parameters.String(
            "Port to shoot at",
            default=80,
            required=True
        )
        generate_requests_in_plan_creator = sdk2.parameters.Bool(
            "generate_requests_in_plan_creator (if plan_creator is None)",
            default=False
        )

        with generate_requests_in_plan_creator.value[True]:
            prefix = sdk2.parameters.String(
                "Prefix, will be added in every request",
                default=""
            )
            shooting_delay = sdk2.parameters.String(
                "delay of shooting in microseconds",
                name="shooting_delay",
                default="1000",
            )
            handlers_whitelist = sdk2.parameters.List(
                "If not empty only handlers from whitelist will be shot, handlers should start with /",
            )
            handlers_blacklist = sdk2.parameters.List(
                "handlers from blacklist will not be shot, handlers should start with /",
            )
            max_request_number = sdk2.parameters.String(
                "maximum requests number",
                name="max_request_number",
                default="10000"
            )
            yt_logs_limit = sdk2.parameters.String(
                "number of rows in YT table with logs to read",
                name="yt_logs_limit",
                default="1000000"
            )
            raw_requests = sdk2.parameters.Resource(
                "Resource with requests from YT logs",
                name="raw requests",
                resource_type=VhDolbilkaRawRequests,
                default=None,
            )
            yt_token_vault = sdk2.parameters.String(
                "YT_TOKEN vault name",
                name="yt_token_vault",
                default="yt_token_for_testenv",
                required=True,
            )
            additional_query_params = sdk2.parameters.List(
                "Additional query params, will be in request as &param1&param2",
                default={}
            )

        with sdk2.parameters.Output():
            shooting_dump = sdk2.parameters.Resource(
                "shooting dump",
                resource_type=DolbilkaDumpResult,
            )
            is_shooting_success = sdk2.parameters.Bool(
                "is shooting success"
            )
            shooting_stats = sdk2.parameters.Resource(
                "shooting statistics",
                resource_type=VhDolbilkaShootingStats,
            )
            request_number = sdk2.parameters.Integer(
                "request number"
            )

    def on_execute(self):
        self._start_shoot(
            port=self.Parameters.custom_port,
            host=self.Parameters.custom_host,
        )

    def _start_shoot(self, port, host):
        request_number = None
        plan = None
        if self.Parameters.plan_creator is None:
            with self.memoize_stage.create_children:
                logging.info("Creating child task VhDolbilkaPlanCreator")
                plan_creator = VhDolbilkaPlanCreator(
                    self,
                    host=self.Parameters.custom_host,
                    port=self.Parameters.custom_port,
                    should_generate_requests=self.Parameters.generate_requests_in_plan_creator,
                    prefix=self.Parameters.prefix,
                    additional_query_params=self.Parameters.additional_query_params,
                    shooting_delay=self.Parameters.shooting_delay,
                    handlers_whitelist=self.Parameters.handlers_whitelist,
                    handlers_blacklist=self.Parameters.handlers_blacklist,
                    max_request_number=self.Parameters.max_request_number,
                    yt_logs_limit=self.Parameters.yt_logs_limit,
                    raw_requests=self.Parameters.raw_requests,
                    yt_token_vault=self.Parameters.yt_token_vault
                ).enqueue()
                self.Context.child_task_id = plan_creator.id
                logging.info("plan_creator.id: %s" % plan_creator.id)

                raise sdk2.WaitTask(
                    [plan_creator],
                    set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
                )

            plan = sdk2.ResourceData(sdk2.Task[self.Context.child_task_id].Parameters.plan)
            request_number = sdk2.Task[self.Context.child_task_id].Parameters.request_number
        else:
            plan_creator = self.Parameters.plan_creator
            plan = sdk2.ResourceData(plan_creator.Parameters.plan)
            if plan_creator.Parameters.request_number is not None and plan_creator.Parameters.request_number > 0:
                request_number = plan_creator.Parameters.request_number
            else:
                logging.info("plan_creator.Parameters.request_number is not defined. Set default request_number = 100")
                request_number = 100

        if request_number <= 0:
            logging.info("request_number <= 0. Set default request_number = 100")
            request_number = 100

        logging.info("request_number: %d", request_number)
        self.Parameters.request_number = request_number

        setattr(self.Context, dolbilka.DolbilkaExecutorRequestsLimit.name, request_number)
        setattr(self.Context, dolbilka.DolbilkaMaximumSimultaneousRequests.name, 1)

        self.Parameters.shooting_dump = DolbilkaDumpResult(self, "dolbilka dump", "result.dump")
        dump_file = sdk2.ResourceData(self.Parameters.shooting_dump).path

        logging.info("Executing started")

        d_executor = dolbilka.DolbilkaExecutor()
        d_executor.mode = dolbilka.DolbilkaExecutorMode.FINGER_MODE
        d_executor.run_session(
            str(plan.path.resolve()),
            dump=str(dump_file),
            save_answers=True,
            host=host,
            port=port,
            circular=False,
        )

        logging.info("Executing done")

        d_dumper = dolbilka.DolbilkaDumper()
        parsed_stats = d_dumper.get_results(str(dump_file), parse_results=True)

        self.Parameters.shooting_stats = VhDolbilkaShootingStats(self, "dolbilka shooting statistics", "shooting_stats.json")
        stats_file = sdk2.ResourceData(self.Parameters.shooting_stats).path

        with open(str(stats_file), 'w') as f:
            f.write(json.dumps(parsed_stats))

        self.Parameters.is_shooting_success = True
        shooting_success = sdk2.ResourceData(DolbilkaShootingSuccess(
            self,
            description="Result of VhDolbilkaShooter (bool)",
            path="is_shooting_success.txt"
        ))
        shooting_success.path.write_bytes("True")
