# -*- coding: utf-8 -*-

import json
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants.constants as consts
import sandbox.common.types.task as ctt
import tarfile
import os
from sandbox.common.types.client import Tag

from sandbox import sdk2
from sandbox import common
from sandbox.projects.common import dolbilka
from sandbox.projects.common.build.arcadia_project_misc import get_arcadia_project_base_target_params
from sandbox.projects.common.utils import sync_last_stable_resource
from sandbox.projects.geobase.GeodataTreeLingStable.resource import GEODATA_TREE_LING_STABLE
from sandbox.projects.resource_types import (
    BUILD_OUTPUT,
    VH_MESSAGE_TEMPLATES,
    VH_LUA_TEMPLATES,
)
from sandbox.projects.vh.frontend import (
    DolbilkaDumpResult,
    SQL_READER_CONFIG,
    VHDatabaseSnapshot,
    VhFrontendSqlTables,
)
from sandbox.projects.vh.frontend.stand_builder.daemon import Daemon
from sandbox.projects.vh.frontend.dolbilka_plan_creator import VhDolbilkaPlanCreator
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.path import Path

ARCADIA_BINARIES_PATH = Path("yabs/vh/frontend/test/back2back")
ARCHIVE_BINARIES_PATH = Path("video-hosting-package/linux")
ARCHIVE_BINARIES_PATH_MKDB = Path("abs/vh/frontend/package/linux")
HAHN_YT_PROXY = 'hahn'

SQL_BASES_TYPES = [
    "tmpl",
    "const",
    "region",
]

YT_BASES_TYPES = [
    "cinf",
]


class VhFrontendProfileResult(sdk2.Resource):
    executable = False


class VhFrontendGetProfile(sdk2.Task):
    """
        Get VH frontend profile stats
    """

    class Requirements(sdk2.Requirements):
        privileged = True
        client_tags = Tag.INTEL_E5_2650 & Tag.LXC & Tag.GENERIC
        execution_space = 10 * 1024
        required_ram = 4 * 1024

    class Parameters(sdk2.Parameters):
        plan_creator = sdk2.parameters.Task(
            "Task that created shooting plan",
            task_type=VhDolbilkaPlanCreator,
            required=True,
        )

        arcadia_url = sdk2.parameters.ArcadiaUrl(
            "Arcadia url",
            required=True,
        )
        template_resource = sdk2.parameters.Resource(
            "Message templates resource",
            resource_type=VH_MESSAGE_TEMPLATES,
            default=None,
        )

        template_resource_lua = sdk2.parameters.Resource(
            "Lua templates resource",
            resource_type=VH_LUA_TEMPLATES,
            default=None,
        )

        mysql_tables = sdk2.parameters.Resource(
            "MySQL DB archive",
            resource_type=VhFrontendSqlTables,
            required=True
        )

        yt_index_table_path = sdk2.parameters.String(
            "YT path to index table",
            default="//home/videoindex/full/vh/prevdata/Index",
        )

    def on_execute(self):
        with self.memoize_stage.build_targets:
            self._ya_make()

        subtasks = self.find(status=ctt.Status.Group.SUCCEED)
        if not subtasks.count:
            raise common.errors.TaskFailure("ya make failed")

        binaries = self._get_bin_path(subtasks.first())
        bases = self._prepare_bases(binaries)

        self._run_tests(binaries, bases)

        try:
            vh_profile_result = sdk2.ResourceData(VhFrontendProfileResult(
                self, 'Vh frontend profile result', 'profile'
            ))
            vh_profile_result.ready()
        except:
            self._logger.info("Unable to open 'profile' directory")

    def _apt_update(self):
        with sdk2.helpers.ProcessLog(self, "apt_get_update") as pl:
            sp.check_call(
                ["sudo", "apt-get", "update"],
                stdout=pl.stdout,
                stderr=pl.stdout
            )

    def _wake_up_mysql(self):
        with sdk2.helpers.ProcessLog(self, "mysql_install") as pl:
            self._apt_update()

            packages = [
                "mysql-client-5.5",
                "mysql-server-5.5",
            ]
            sp.check_call(
                ["sudo", "apt-get", "-y", "--force-yes", "install"] + packages,
                stdout=pl.stdout,
                stderr=pl.stdout
            )

        with sdk2.helpers.ProcessLog(self, "mysql_set_up") as pl:
            MYSQL_CREATE_DBS = \
                "create database if not exists vh; " \
                "create database if not exists yabsdb;"

            sp.check_call(
                ["mysql", "-v", "-u", "root", "-e", MYSQL_CREATE_DBS],
                stdout=pl.stdout,
                stderr=pl.stdout,
            )

            mysql_tables = sdk2.ResourceData(self.Parameters.mysql_tables)
            with tarfile.open(str(mysql_tables.path)) as tables_tar:
                tables_tar.extractall("/var/lib")

            sp.check_call(
                ["chown", "-R", "mysql:mysql", "/var/lib/mysql"]
            )

            sp.check_call(
                ["mysql", "-v", "-u", "root", "vh", "-e", "show tables;"],
                stdout=pl.stdout,
                stderr=pl.stdout,
            )
            sp.check_call(
                ["mysql", "-v", "-u", "root", "yabsdb", "-e", "show tables;"],
                stdout=pl.stdout,
                stderr=pl.stdout,
            )

    def _sql_reader_config(self):
        SQL_READER_CONFIG["password"] = sdk2.Vault.data("vh_mkdb_password")
        SQL_READER_CONFIG["sandbox_token"] = sdk2.Vault.data("vh_mkdb_sandbox_token")
        if self.Parameters.mysql_tables is not None:
            SQL_READER_CONFIG["hosts"] = ["localhost"]
            SQL_READER_CONFIG["user"] = "root"
            SQL_READER_CONFIG["password"] = ""

        if self.Parameters.template_resource is not None:
            SQL_READER_CONFIG["template_resource_id"] = self.Parameters.template_resource.id
        if self.Parameters.template_resource_lua is not None:
            SQL_READER_CONFIG["lua_template_resource_id"] = self.Parameters.template_resource_lua.id

        return json.dumps(SQL_READER_CONFIG)

    def _prepare_bases(self, binaries):
        if self.Parameters.mysql_tables is not None:
            self._wake_up_mysql()
        bases = sdk2.ResourceData(
            VHDatabaseSnapshot(
                self,
                description="DB snapshot",
                path="bases",
            ),
        )
        bases.path.mkdir(parents=True)
        self._generate_bases(
            binaries=binaries,
            bases=bases.path,
        )
        bases.ready()
        return bases.path

    def _generate_bases(self, binaries, bases):
        sql_reader_config = self._sql_reader_config()
        for sql_base_type in SQL_BASES_TYPES:
            with sdk2.helpers.ProcessLog(self, "sql_reader_" + sql_base_type) as pl:
                sql_reader = sp.Popen(
                    [
                        str(binaries / "sql_reader"),
                        "--sandbox-run",
                        "--type", sql_base_type
                    ],
                    stdin=sp.PIPE,
                    stdout=sp.PIPE,
                    stderr=pl.stdout,
                )
                raw_data = sql_reader.communicate(sql_reader_config)[0].strip()
            with sdk2.helpers.ProcessLog(self, "mkdb_" + sql_base_type) as pl:
                mkdb = sp.Popen(
                    [str(binaries / "mkdb"), str(bases / ("flat." + sql_base_type)), sql_base_type],
                    stdin=sp.PIPE,
                    stderr=pl.stdout,
                )
                mkdb.communicate(raw_data)
        geobase_path = sync_last_stable_resource(GEODATA_TREE_LING_STABLE)
        self.__geobase_path = os.path.join(os.getcwd(), geobase_path)
        for yt_base_type in YT_BASES_TYPES:
            with sdk2.helpers.ProcessLog(self, "ytdb_" + yt_base_type) as pl:
                sp.Popen(
                    [
                        str(binaries / "ytdb"),
                        self.Parameters.yt_index_table_path,
                        str(bases / ("flat." + yt_base_type)),
                        HAHN_YT_PROXY,
                        sdk2.Vault.data(self.owner, "yt_token"),
                        self.__geobase_path,
                    ],
                    stdin=sp.PIPE,
                    stdout=sp.PIPE,
                    stderr=pl.stdout,
                ).wait()

    def _run_tests(self, binaries, bases):
        with sdk2.helpers.ProcessLog(self, "vh_daemon") as pl:
            with Daemon(
                self,
                binary_path=str(binaries / "daemons"),
                bases_path=str(bases),
                fast_config_path="./fake",
                geodata_file_path=self.__geobase_path,
                stdout=pl.stdout,
                logger=pl.logger,
                run_profile=True
            ) as daemon_port:
                self._start_shoot(daemon_port)

    def _start_shoot(self, daemon_port, host="localhost"):
        plan_creator = self.Parameters.plan_creator
        request_number = plan_creator.Parameters.request_number
        plan = sdk2.ResourceData(plan_creator.Parameters.plan)

        setattr(self.Context, dolbilka.DolbilkaOutputLenval32.name, True)
        setattr(self.Context, dolbilka.DolbilkaMaximumSimultaneousRequests.name, 1)
        setattr(self.Context, dolbilka.DolbilkaExecutorRequestsLimit.name, request_number)

        dolbilka_dump_result = DolbilkaDumpResult(self, "dolbilka dump", "result.dump")
        d_executor = dolbilka.DolbilkaExecutor()
        d_executor.run_session(
            str(plan.path),
            dump=str(dolbilka_dump_result.path),
            save_answers=True,
            host=host,
            port=daemon_port,
        )

    def _ya_make(self):
        waited_statuses = set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK))
        raise sdk2.WaitTask(
            self._create_ya_make_subtask(),
            waited_statuses,
            wait_all=True
        )

    def _get_bin_path(self, ya_make_task):
        build_output_queue = BUILD_OUTPUT.find(task=ya_make_task)
        if not build_output_queue.count:
            raise common.errors.TaskFailure("Can't find built targets. Failing")
        build_output = build_output_queue.first()
        return sdk2.ResourceData(build_output).path / ARCADIA_BINARIES_PATH

    def _create_ya_make_subtask(self):
        task_id = self.server.task({
            "children": True,
            "context": {
                build_params.ArcadiaUrl.name: self.Parameters.arcadia_url,
                build_params.BuildSystem.name: "ya",
                build_params.CheckReturnCode.name: True,
                build_params.CheckoutParameter.name: True,
                build_params.ClearBuild.name: False,
                build_params.TestParameter.name: True,
                build_params.BuildType.name: 'Profile',
                consts.BUILD_TYPE_KEY: 'Profile',
                get_arcadia_project_base_target_params().TargetsParameter.name: str(ARCADIA_BINARIES_PATH),
            },
            "description": "Building VH server",
            "notification": [],
            "owner": self.owner,
            "priority": {
                "class": self.Parameters.priority.cls,
                "subclass": self.Parameters.priority.cls,
            },
            "type": "YA_MAKE",
        })["id"]
        self.server.batch.tasks.start.update([task_id])
        return task_id
