# -*- coding: utf-8 -*-

import datetime
import hashlib
import logging
import os
import time

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common import utils
from sandbox.projects.vh.frontend import (
    VhIndexesDumperBinary,
    VhFlatCinfBase,
)
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
# from sandbox.sdk2.path import Path


logging.basicConfig(
    format="[%(asctime)s %(name)s %(levelname)s] %(message)s",
    level=logging.INFO
)


def md5(fname):
    hash_md5 = hashlib.md5()
    with open(fname, "rb") as f:
        for chunk in iter(lambda: f.read(16384), b""):
            hash_md5.update(chunk)
    return hash_md5.hexdigest()


class VhFrontendGenerateBases(sdk2.Task):
    class Requirements(sdk2.Requirements):
        # privileged = True
        client_tags = Tag.INTEL_E5_2650 & Tag.LXC & Tag.GENERIC
        disk_space = 40 * 1024
        ram = 40 * 1024
        environments = [
            PipEnvironment("yandex-yt", "0.9.26"),
        ]

    class Parameters(sdk2.Parameters):
        yt_index_table_path = sdk2.parameters.String(
            "YT path to index table",
            default="//home/video-hosting/index/base_generation/prevdata",
            required=True,
        )
        yt_proxy = sdk2.parameters.String(
            "YT proxy to read tables",
            default="arnold",
            required=True,
        )
        yt_token = sdk2.parameters.String(
            "YT token vault name",
            name="yt_token",
            default="yt_token_for_basegen",
            required=True,
        )
        indexes_dumper = sdk2.parameters.Resource(
            "Resource with indexes_dump binary",
            resource_type=VhIndexesDumperBinary,
            default=None,
        )
        with sdk2.parameters.Output():
            cinf_base = sdk2.parameters.Resource(
                "cinf base dump",
                resource_type=VhFlatCinfBase,
            )

    def on_execute(self):
        modification_timestamp_from_yt = self.get_modification_time_from_yt()
        logging.info("Last modify time for YT ContentInfo: " + str(modification_timestamp_from_yt))
        modification_timestamp_from_last_resource = self.get_modification_time_from_last_resource()
        logging.info("Last modify time for SB ContentInfo: " + str(modification_timestamp_from_last_resource))
        if modification_timestamp_from_yt <= modification_timestamp_from_last_resource:
            return

        base_name = self.Parameters.yt_proxy + ".flat.cinf"
        self.Parameters.cinf_base = VhFlatCinfBase(self, "cinf base dump", base_name)
        bases_path = self.Parameters.cinf_base.path.as_posix()
        logging.info("Binary path: " + os.path.abspath(self.get_indexes_dumper_path()))
        with sdk2.helpers.ProcessLog(self, "indexes_dumper_" + self.Parameters.yt_proxy):
            dumper = sp.Popen(
                [
                    os.path.abspath(self.get_indexes_dumper_path()),
                    "--server", self.Parameters.yt_proxy,
                    "--static", self.Parameters.yt_index_table_path,
                    "--flat", bases_path,
                    "--version", self.get_max_unix_time(),
                ],
                stdin=sp.PIPE,
                stdout=sp.PIPE,
                stderr=sp.PIPE,
                env={"YT_TOKEN": self.get_yt_token()}
            )

            out, err = dumper.communicate()
            logging.info('message ' + str(out))
            logging.info('error ' + str(err))

            if dumper.returncode != 0:
                raise RuntimeError("Unable to generate base file")

        mb_size = os.path.getsize(bases_path) >> 20
        if mb_size < 2000:
            raise RuntimeError("Index file is too small (mb): " + str(mb_size))

        logging.info("Base File md5: " + md5(bases_path))
        utils.set_resource_attributes(self.Parameters.cinf_base, {"modification_time": modification_timestamp_from_yt})

    def get_yt_token(self):
        return sdk2.Vault.data(self.Parameters.yt_token)

    def has_new_base(self):
        modification_timestamp_from_yt = self.get_modification_time_from_yt()
        logging.info("Last modify time for YT ContentInfo: " + str(modification_timestamp_from_yt))
        modification_timestamp_from_last_resource = self.get_modification_time_from_last_resource()
        logging.info("Last modify time for sb ContentInfo: " + str(modification_timestamp_from_last_resource))
        return modification_timestamp_from_yt != modification_timestamp_from_last_resource

    def get_modification_time_from_yt(self):
        from yt.wrapper import YtClient, ypath_join
        client = YtClient(proxy=self.Parameters.yt_proxy, token=self.get_yt_token())
        yt_table = ypath_join(self.Parameters.yt_index_table_path, "ContentInfo")
        modification_time = str(client.get("{table}/@modification_time".format(table=yt_table)))
        modification_timestamp = datetime.datetime.strptime(modification_time, '%Y-%m-%dT%H:%M:%S.%fZ').strftime('%s')
        return int(modification_timestamp)

    def get_modification_time_from_last_resource(self):
        try:
            resource = self.sync_last_resource(VhFlatCinfBase)
            return resource.modification_time
        except:
            return 0

    def sync_last_resource(self, resource_type):
        # FIXME: read actual resource_id from YT and sync it
        resource_id = utils.get_and_check_last_resource_with_attribute(resource_type, attr_name="modification_time")  # .id?
        return utils.sync_resource(resource_id)

    def get_indexes_dumper_path(self):
        if self.Parameters.indexes_dumper is not None:
            return str(sdk2.ResourceData(self.Parameters.indexes_dumper).path)
        else:
            return utils.sync_last_stable_resource(VhIndexesDumperBinary)

    def get_max_unix_time(self):
        max_unix_time = str(int(time.time()))
        logging.info("Index max_unix_time: " + max_unix_time)
        return max_unix_time
