# -*- coding: utf-8 -*-

import logging
import json

import sandbox.common.types.task as ctt

from sandbox.sandboxsdk import environments
from os.path import join as join_path, realpath, dirname
from sandbox.sdk2.helpers import subprocess as sp
from sandbox import sdk2
from sandbox import common


class VhStat(sdk2.Task):
    class Requirements(sdk2.Requirements):
        environments = (environments.PipEnvironment('yandex-yt', use_wheel=True),)

    class Parameters(sdk2.Task.Parameters):
        task_name = sdk2.parameters.String(
            "Уникальное имя задачи",
            name="task_name",
            default="",
            required=True,
        )

        yt_cluster = sdk2.parameters.String(
            "YT cluster (i.e. hahn)",
            name="yt_cluster",
            default="hahn",
            required=True,
        )

        yt_token = sdk2.parameters.String(
            "YT token vault name",
            name="yt_token",
            default="yt_token",
            required=True,
        )

        yql_token = sdk2.parameters.String(
            "YQL token vault name",
            name="yql_token",
            default="yql_token",
            required=True,
        )

        yt_table_def = sdk2.parameters.String(
            "YT table defenition, yson format",
            name="yt_table_def",
            default="",
            required=True,
            multiline=True,
        )

        yt_result_table_name = sdk2.parameters.String(
            "Path to result YT table",
            name="yt_result_table_name",
            default="",
            required=True,
        )

        yql_request = sdk2.parameters.String(
            "YQL запрос",
            name="yql_request",
            default="",
            required=True,
            multiline=True,
        )

        request_log_path = sdk2.parameters.String(
            "Request logs path in YT",
            name="request_log_path",
            default="",
            required=True,
        )
        last_date_time = sdk2.parameters.String(
            "Дата (и если необходимо время) последнего успешного расчета",
            name="last_date",
            default="",
            required=True,
        )
        stop_date_time = sdk2.parameters.String(
            "Дата (и если необходимо время) на которой надо остановиться в рассчетах",
            name="stop_date",
            default="",
            required=False,
        )

        version = sdk2.parameters.String(
            "Номер версии кода",
            name="version",
            default="1",
            required=True,
        )

    def on_execute(self):

        tables = self.get_tables()

        last_date_time = self.Parameters.last_date_time
        if last_date_time == "":
            raise common.errors.TaskFailure("last_date_time not defined")

        find_task = sdk2.Task.find(VhStat, status=(ctt.Status.Group.SUCCEED), input_parameters={
            'task_name': self.Parameters.task_name,
            'version': self.Parameters.version
        }).order(-sdk2.Task.id).first()

        if find_task and find_task.Context.last_date_time:
            last_date_time = find_task.Context.last_date_time

        logging.info(last_date_time)

        unprocessed_tables = filter(lambda x: x > last_date_time, tables)
        if self.Parameters.stop_date_time != "":
            unprocessed_tables = filter(lambda x: x <= self.Parameters.stop_date_time, unprocessed_tables)

        logging.info("Found logs: %s" % ", ".join(unprocessed_tables))

        if len(unprocessed_tables) > 1:  # Crutch for strm
            date_time_to_process = unprocessed_tables[0]
            self.save_params(date_time_to_process)
            self.make_yql_work(date_time_to_process)

            self.Context.last_date_time = date_time_to_process

    def save_params(self, date_time_to_process):
        params = {}

        params["yt_cluster"] = self.Parameters.yt_cluster
        params["yt_token"] = sdk2.Vault.data(self.Parameters.yt_token)

        params["yql_token"] = sdk2.Vault.data(self.Parameters.yql_token)
        params["yql_request"] = self.Parameters.yql_request

        params["date_time"] = date_time_to_process

        params["yt_table_def"] = self.Parameters.yt_table_def
        params["yt_result_table_name"] = self.Parameters.yt_result_table_name

        with open("params", "w") as f:
            json.dump(params, f)

    def make_yql_work(self, date_time):

        yql_token = sdk2.Vault.data(self.Parameters.yql_token)
        logging.info(yql_token)

        with environments.VirtualEnvironment(use_system=True) as venv:
            venv.pip("pip setuptools==33.1.1")
            requirements = (
                "yandex-yt=={ver}".format(ver="0.7.38.post0"),
                "yandex-yt-yson-bindings-skynet=={ver}".format(ver="0.2.25.post1"),
            )
            venv._VirtualEnvironment__run_cmd(
                '{python} -us {pip} install -i {pypi} -vvv {req}'.format(
                    python=venv.executable,
                    pip=venv._VirtualEnvironment__pip_wrapper,
                    pypi='https://pypi.yandex-team.ru/simple/',
                    req=" ".join(requirements),
                ), log_prefix='vh_statistic_install'
            )

            environments.PipEnvironment('yql', venv=venv).prepare()
            environments.PipEnvironment('pyyaml', venv=venv).prepare()

            script_path = join_path(dirname(realpath(__file__)), 'calculate_statistics.py')
            with sdk2.helpers.ProcessLog(self, "vh_statistics") as pl:
                sp.check_call([venv.executable, script_path], stdout=pl.stdout, stderr=pl.stdout)

    def get_tables(self):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        yt_cluster = self.Parameters.yt_cluster
        client = YtClient(yt_cluster, yt_token)
        tables = sorted(client.list(self.Parameters.request_log_path))
        logging.info("Found logs: %s" % ", ".join(tables))
        return tables
