# -*- coding: utf-8 -*-
"""
The hodgepodge of function and classes
"""
import yaml
import time
import re
from random import randint
import requests

import yt.wrapper as yt
from yt.wrapper import YtClient
import ylogging
import consts
import yt.yson as yson

logger = ylogging.get_logger()


class Retry(object):
    def __init__(self, exception, attempts=3, max_sleep=120, verbose=True):
        self.exception = exception
        self.attempts = attempts
        self.max_sleep = max_sleep
        self.verbose = verbose
        self.func = None  # Calm down pylint

    def __call__(self, func):
        self.func = func
        return self._retrier

    def _retrier(self, *args, **kwargs):
        cur_attempt = 0
        step = self.max_sleep / self.attempts
        while True:
            cur_attempt += 1
            try:
                return self.func(*args, **kwargs)
            except self.exception as e:
                if cur_attempt < self.attempts:
                    if self.verbose:
                        logger.error("Attempt %d failed with error: %s", cur_attempt, e.message)
                    if self.max_sleep > 0:
                        time_to_sleep = randint(cur_attempt * step, self.max_sleep)
                        logger.info("time to sleep %d" % time_to_sleep)
                        time.sleep(time_to_sleep)
                else:
                    raise
            except:
                if cur_attempt < self.attempts:
                    if self.verbose:
                        logger.error("Attempt %d failed with error: %s", cur_attempt, "unhandled exception")
                    if self.max_sleep > 0:
                        time_to_sleep = randint(cur_attempt * step, self.max_sleep)
                        logger.info("time to sleep %d" % time_to_sleep)
                        time.sleep(time_to_sleep)
                else:
                    raise


def yt_connect(proxy, token=None):
    config = {
        "proxy": {
            "url": proxy
        },
        "token": token,
        "tabular_data_format": yt.JsonFormat(),
    }
    ytc = YtClient(config=config)
    for m in "select_rows", "delete_rows", "insert_rows":
        setattr(ytc, m, Retry(yt.errors.YtError, attempts=10)(getattr(ytc, m)))
    return ytc


def _yt_reshard_table(ytc, table_name, reshard_rule):
    count = reshard_rule["count"]
    min_val = reshard_rule.get("min", 0)
    max_val = reshard_rule.get("max", (1 << 64))

    delta = max_val - min_val
    pivot_keys = [[]]
    for i in xrange(1, count):
        pivot_keys.append([yson.YsonUint64(min_val + (i * delta) // count)])

    ytc.reshard_table(table_name, pivot_keys)


def _yt_await_tablet_state(ytc, table_name, state):
    while not all(x["state"] == state for x in ytc.get(table_name + "/@tablets")):
        time.sleep(0.25)


def yt_mount_table(ytc, table_name):
    ytc.mount_table(table_name)
    _yt_await_tablet_state(ytc, table_name, "mounted")


def yt_unmount_table(ytc, table_name):
    ytc.unmount_table(table_name)
    _yt_await_tablet_state(ytc, table_name, "unmounted")


def yt_create_table(ytc, table_name, table_def, drop_if_exists=False,
                    table_def_format="yaml"):
    """Create dynamic yt table. Definition is based on yabs-yt-api format

    NOTE: Don't change already existing tables (may be changed in the future)

    table_name - full name of table (with path, no default path supported)
    table_def - table schema
    """
    if ytc.exists(table_name):
        if drop_if_exists:
            yt_unmount_table(ytc, table_name)
            ytc.remove(table_name)
        else:
            return

    if hasattr(table_def, "read"):
        table_def = table_def.read()

    if isinstance(table_def, str):
        if table_def_format == "yaml":
            table_def = yaml.load(table_def)
        elif table_def_format == "yson":
            from yt import yson
            table_def = yson.loads(table_def)

    if not isinstance(table_def, dict):
        raise Exception("table_def must be file, string or dict")

    table_def["attributes"]["dynamic"] = True
    ytc.create_table(table_name, attributes=table_def["attributes"])

    reshard_rule = table_def.get("reshard_rule")
    if reshard_rule:
        _yt_reshard_table(ytc, table_name, reshard_rule)

    yt_mount_table(ytc, table_name)


def yt_drop_table(ytc, table_name):
    tablets = ytc.get_attribute(table_name, "tablets", default=[])
    if tablets:
        yt_unmount_table(ytc, table_name)
    ytc.remove(table_name)


def yt_freeze_table(ytc, table_name):
    ytc.freeze_table(table_name)
    _yt_await_tablet_state(ytc, table_name, "frozen")


def yt_unfreeze_table(ytc, table_name):
    ytc.unfreeze_table(table_name)
    _yt_await_tablet_state(ytc, table_name, "mounted")


def yt_clone_table(ytc, source, destination):
    yt_freeze_table(ytc, source)
    try:
        ytc.copy(source, destination)
    finally:
        yt_unfreeze_table(ytc, source)
    yt_mount_table(ytc, destination)


def get_yt_version(ytc):
    url = 'http://' + ytc.config['proxy']['url'] + '/api/v3/get_version'
    token = ytc.config.get('token')
    headers = {'Authorization': 'OAuth ' + token} if token else None
    response = requests.get(url, headers=headers)
    response.raise_for_status()
    raw_version = response.content
    m = re.match(r'"(?:0.)?(\d+)\.(\d+)', raw_version)
    return '_'.join(m.groups())


def get_all_logs(ytc, log_path, verbose=True):
    result = {}
    if ytc.exists(log_path):
        for log_type, logs in ytc.get(log_path).iteritems():
            log_list = result.setdefault(log_type, [])
            for log in logs.keys():
                full_log_name = "/".join([log_path, log_type, log])
                if verbose:
                    attrs = ytc.get(full_log_name + "/@")
                    logger.info("Found ready log '%s', %d rows, %d bytes", full_log_name, attrs["row_count"], attrs["uncompressed_data_size"])
                log_list.append(full_log_name)
    else:
        logger.info("Path %s does not exist", log_path)
    return result


def get_log_list(ytc, log_path, log_type, verbose=True):
    log_list = []
    log_path += "/" + log_type
    if ytc.exists(log_path):
        for log in ytc.list(log_path):
            full_log_name = log_path + "/" + log
            if verbose:
                attrs = ytc.get(full_log_name + "/@")
                logger.info("Found ready log '%s', %d rows, %d size", full_log_name, attrs["row_count"], attrs["uncompressed_data_size"])
            log_list.append(full_log_name)
    else:
        logger.info("Path %s does not exist", log_path)
    return log_list


class YtLock(object):
    def __init__(self, ytc, lock_name, release_type=""):
        if release_type == consts.ReleaseTypeEnum.PREPROD:
            lock_name = "preprod." + lock_name
        self.path = consts.YT_LOCK_PATH + "/" + lock_name
        ytc.create("int64_node", self.path, ignore_existing=True, recursive=True)

    def get_lock_count(self, ytc):
        locks = ytc.get_attribute(self.path, "locks")
        return sum(1 for lk in locks if lk["mode"] == "exclusive")

    def lock(self, ytc, wait_for=600000):
        ytc.lock(self.path, mode="exclusive", waitable=True, wait_for=wait_for)


class AppControl(object):
    def __init__(self, ytc):
        self._app_control = {}
        if not ytc.exists(consts.YT_APP_CONTROL_PATH):
            ytc.create("map_node", consts.YT_APP_CONTROL_PATH, recursive=True, ignore_existing=True)
        else:
            for k, v in ytc.get(consts.YT_APP_CONTROL_PATH).items():
                v = v.lower()
                if v in ('enabled', 'paused', 'disabled'):
                    self._app_control[k] = v
                else:
                    logger.warn("Wrong app control value '{}' for '{}'".format(v, k))
                    # Treat wrong value as 'paused' as the most safe and recoverable state
                    self._app_control[k] = 'paused'

    def is_collector_can_run(self, collector_name):
        return self._app_control.get(collector_name, 'enabled') == 'enabled'

    def is_collector_enabled(self, collector_name):
        return self._app_control.get(collector_name, 'enabled') != 'disabled'
