import os

from sandbox import sdk2
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process

from sandbox.projects.VideoSearch.video_resource_types import VIDEO_CATEG_INFO_STORAGE
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import utils
from sandbox.projects.common.mediasearch import ban as mediaban
from sandbox.projects.common.nanny import auto_deploy

import sandbox.projects.video.quality.recommender.VideoRecommenderTestMiddleResources as video_test

import sandbox.common.types.task as ctt


_VIDEO_CATEG_INFO_STORAGE_ID = "video_categ_info_storage_id"
_VIDEO_RECOMMENDER_TEST_MIDDLE_RESOURCES_TASK_ID = "video_recommender_test_middle_resources_task_id"
_QUERIES_LIMIT = "queries_limit"
_MAX_FAIL_RATE = "max_fail_rate"
_MAX_NOT_RESPONDED_REQIDS_PERCENT = "max_not_responded_reqids_percent"


class YTServerParameter(parameters.SandboxStringParameter):
    name = "yt_server"
    description = "YT server"
    required = True
    default_value = "arnold"


class YtCategInfoStorage(parameters.SandboxStringParameter):
    name = "yt_videohub_categ_info_storage"
    description = "YT videohub categ info storage file"
    required = True
    default_value = "//home/videoindex/recommender/files/index/categ_info_storage"


class VideoReleaseCategInfoStorage(auto_deploy.AutoNannyDeployTask, mediaban.VideoBaseReleaseBanTask):
    """
        Builds categ_info_storage for Yandex.Video service
    """

    type = "VIDEO_RELEASE_CATEG_INFO_STORAGE"

    input_parameters = (
        YTServerParameter,
        YtCategInfoStorage,
    ) + mediaban.VideoBaseReleaseBanTask.input_parameters + (
        video_test.VideoRecommenderTestMiddleResources.Parameters.queries_limit,
        video_test.VideoRecommenderTestMiddleResources.Parameters.max_fail_rate,
        video_test.VideoRecommenderTestMiddleResources.Parameters.max_not_responded_reqids_percent,
    )

    release_subject = "video/middle/videohub-categ_info_storage-{timestamp}"
    release_comment = "video fast categ info storage"
    release_resources = (
        VIDEO_CATEG_INFO_STORAGE,
    )

    def on_execute(self):
        if _VIDEO_CATEG_INFO_STORAGE_ID not in self.ctx:
            mediaban.VideoBaseReleaseBanTask.on_execute(self)
            return

        if _VIDEO_RECOMMENDER_TEST_MIDDLE_RESOURCES_TASK_ID not in self.ctx:
            test_ctx = {
                'categ_info_storage_id': self.ctx[_VIDEO_CATEG_INFO_STORAGE_ID],
                # Next params're passed here from the _test_ban.
                _QUERIES_LIMIT: self.ctx[_QUERIES_LIMIT],
                _MAX_FAIL_RATE: self.ctx[_MAX_FAIL_RATE],
                _MAX_NOT_RESPONDED_REQIDS_PERCENT: self.ctx[_MAX_NOT_RESPONDED_REQIDS_PERCENT],
            }

            test_task_class = sdk2.Task['VIDEO_RECOMMENDER_TEST_MIDDLE_RESOURCES']
            test_task = test_task_class(test_task_class.current, **test_ctx)
            test_task.enqueue()

            self.ctx[_VIDEO_RECOMMENDER_TEST_MIDDLE_RESOURCES_TASK_ID] = test_task.id

            self.wait_tasks([test_task], tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK), True)

        test_video_categ_info_storage_task = sdk2.Task.find(
            id=self.ctx[_VIDEO_RECOMMENDER_TEST_MIDDLE_RESOURCES_TASK_ID]
        ).first()

        if test_video_categ_info_storage_task.status not in ctt.Status.Group.SUCCEED:
            error_msg = 'VIDEO_RECOMMENDER_TEST_MIDDLE_RESOURCES task failed.'
            raise errors.SandboxTaskFailureError(error_msg)

    def get_stable_services(self):
        return [self._SERVICE_ID]

    def get_nanny_oauth_token(self):
        return self.get_vault_data('VIDEO-ROBOT', 'robot-video-crawl-nanny-oauth')

    def _yt(self, output, cmd, *args):
        yt_tool = self._tool(resource_types.VIDEO_YT_PYTHON_EXECUTABLE)
        yt_args = (
            yt_tool,
            "--proxy", self.ctx[YTServerParameter.name],
        )

        return process.run_process(
            yt_args + (cmd,) + args,
            environment={"YT_TOKEN": self.get_vault_data('VIDEODEV', 'yt_token')},
            outputs_to_one_file=False,
            log_prefix="yt.{}".format(cmd),
            timeout=600,
            wait=False,
            stdout=output
        )

    def _read_yt_file(self, file_path, dst_path):
        yt_exists = self._yt(None, "exists", file_path)
        if yt_exists.wait():
            raise errors.SandboxTaskFailureError("Failed to check existence %s" % (file_path))

        exists = False
        with open(yt_exists.stdout_path, 'r') as f:
            exists = str.strip(f.readline()) == "true"

        if not exists:
            return False

        with open(dst_path, "w") as output:
            yt_read_file = self._yt(output, "read-file", "--path", file_path)
            if yt_read_file.wait():
                raise errors.SandboxTaskFailureError("Failed to read {}".format(file_path))
        return True

    def _build_ban(self):
        local_path = self.abs_path("./categ_info_storage")
        if not self._read_yt_file(self.ctx[YtCategInfoStorage.name], local_path):
            return 0

        has_changes = self._update_resource(VIDEO_CATEG_INFO_STORAGE, path=local_path)
        if not has_changes:
            return 0

        return os.stat(local_path).st_size

    def _test_ban(self, build_task_id):
        video_categ_info_storage_id = \
            apihelpers.get_task_resource_id(build_task_id, VIDEO_CATEG_INFO_STORAGE)

        queries_limit = video_test.VideoRecommenderTestMiddleResources.Parameters.queries_limit
        max_fail_rate = video_test.VideoRecommenderTestMiddleResources.Parameters.max_fail_rate
        max_not_responded_reqids_percent = \
            video_test.VideoRecommenderTestMiddleResources.Parameters.max_not_responded_reqids_percent

        test_ctx = {
            _VIDEO_CATEG_INFO_STORAGE_ID: video_categ_info_storage_id,
            _QUERIES_LIMIT: utils.get_or_default(self.ctx, queries_limit),
            _MAX_FAIL_RATE: utils.get_or_default(self.ctx, max_fail_rate),
            _MAX_NOT_RESPONDED_REQIDS_PERCENT: utils.get_or_default(self.ctx, max_not_responded_reqids_percent),
            'do_not_restart': True,
        }

        test_task = self.create_subtask(
            task_type=self.type,
            description=self.descr,
            inherit_notifications=True,
            input_parameters=test_ctx,
            priority=self.priority
        )

        return [test_task.id]

__Task__ = VideoReleaseCategInfoStorage
