import os

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process

from sandbox.projects.VideoSearch.video_resource_types import VIDEO_MIDDLE_RECOMMENDER_MODELS
from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as mediaban
from sandbox.projects.common.nanny import auto_deploy


class YTServerParameter(parameters.SandboxStringParameter):
    name = "yt_server"
    description = "YT server"
    required = True
    default_value = "arnold"


class YTDjMiddleModelsTar(parameters.SandboxStringParameter):
    name = "yt_videohub_dj_models"
    description = "YT videohub dj middle models tar archive"
    required = True
    default_value = "//home/videorecom/recommender/files/dj_middle/dj_middle_models.tar"


class NanyOauthRole(parameters.SandboxStringParameter):
    name = "nany_oauth_role"
    description = "OAuth role to get Nany token"
    required = True
    default_value = 'VIDEO-ROBOT'


class NanyOauthToken(parameters.SandboxStringParameter):
    name = "nany_oauth_token"
    description = "OAuth token name to get for Nany"
    required = True
    default_value = 'robot-video-crawl-nanny-oauth'


class YTProcessTokenRole(parameters.SandboxStringParameter):
    name = "yt_process_token_role"
    description = "Role to extract yt token"
    required = True
    default_value = 'VIDEODEV'


class YTProcessToken(parameters.SandboxStringParameter):
    name = "yt_process_token"
    description = "Token for yt to connect during resource uploading"
    required = True
    default_value = "yt_token"


class VideoReleaseDjMiddleModels(auto_deploy.AutoNannyDeployTask, mediaban.VideoBaseReleaseBanTask):
    """
        Builds dj_middle_recommender models for Yandex.Efir service
    """

    type = "VIDEO_RELEASE_MIDDLE_RECOMMENDER_MODELS"

    input_parameters = (
        YTServerParameter,
        YTDjMiddleModelsTar,
        NanyOauthRole,
        NanyOauthToken,
        YTProcessTokenRole,
        YTProcessToken,
    ) + mediaban.VideoBaseReleaseBanTask.input_parameters

    release_subject = "video/middle/dj-middle-models-{timestamp}"
    release_comment = "video fast dj_middle_models.tar"
    release_resources = (
        VIDEO_MIDDLE_RECOMMENDER_MODELS,
    )

    def get_stable_services(self):
        return [self._SERVICE_ID]

    def get_nanny_oauth_token(self):
        return self.get_vault_data(self.ctx[NanyOauthRole.name], self.ctx[NanyOauthToken.name])

    def _yt(self, output, cmd, *args):
        yt_tool = self._tool(resource_types.VIDEO_YT_PYTHON_EXECUTABLE)
        yt_args = (
            yt_tool,
            "--proxy", self.ctx[YTServerParameter.name],
        )

        return process.run_process(
            yt_args + (cmd,) + args,
            environment={"YT_TOKEN": self.get_vault_data(self.ctx[YTProcessTokenRole.name], self.ctx[YTProcessToken.name])},
            outputs_to_one_file=False,
            log_prefix="yt.{}".format(cmd),
            timeout=600,
            wait=False,
            stdout=output
        )

    def _read_yt_file(self, file_path, dst_path):
        yt_exists = self._yt(None, "exists", file_path)
        if yt_exists.wait():
            raise errors.SandboxTaskFailureError("Failed to check existence %s" % (file_path))

        exists = False
        with open(yt_exists.stdout_path, 'r') as f:
            exists = str.strip(f.readline()) == "true"

        if not exists:
            return False

        with open(dst_path, "w") as output:
            yt_read_file = self._yt(output, "read-file", "--path", file_path)
            if yt_read_file.wait():
                raise errors.SandboxTaskFailureError("Failed to read {}".format(file_path))
        return True

    def _build_ban(self):
        local_path = self.abs_path("./dj_middle_models.tar")
        if not self._read_yt_file(self.ctx[YTDjMiddleModelsTar.name], local_path):
            return 0

        has_changes = self._update_resource(VIDEO_MIDDLE_RECOMMENDER_MODELS, path=local_path)
        if not has_changes:
            return 0

        return os.stat(local_path).st_size

__Task__ = VideoReleaseDjMiddleModels
