import os

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process

from sandbox.projects.VideoSearch.video_resource_types import VIDEO_SUPERTAGS
from sandbox.projects import resource_types
from sandbox.projects.common.mediasearch import ban as mediaban
from sandbox.projects.common.nanny import auto_deploy


class YTServerParameter(parameters.SandboxStringParameter):
    name = "yt_server"
    description = "YT server"
    required = True
    default_value = "arnold"


class YtSuperTags(parameters.SandboxStringParameter):
    name = "yt_videohub_supertags"
    description = "YT videohub supertags file"
    required = True
    default_value = "//home/videoindex/recommender/files/index/videohub_supertags"


class VideoReleaseSuperTags(auto_deploy.AutoNannyDeployTask, mediaban.VideoBaseReleaseBanTask):
    """
        Builds videohub_supertags for Yandex.Video service
    """

    type = "VIDEO_RELEASE_SUPER_TAGS"

    input_parameters = (
        YTServerParameter,
        YtSuperTags,
    ) + mediaban.VideoBaseReleaseBanTask.input_parameters

    release_subject = "video/middle/videohub-supertags-{timestamp}"
    release_comment = "video fast videohub_supertags"
    release_resources = (
        VIDEO_SUPERTAGS,
    )

    def get_stable_services(self):
        return [self._SERVICE_ID]

    def get_nanny_oauth_token(self):
        return self.get_vault_data('VIDEO-ROBOT', 'robot-video-crawl-nanny-oauth')

    def _yt(self, output, cmd, *args):
        yt_tool = self._tool(resource_types.VIDEO_YT_PYTHON_EXECUTABLE)
        yt_args = (
            yt_tool,
            "--proxy", self.ctx[YTServerParameter.name],
        )

        return process.run_process(
            yt_args + (cmd,) + args,
            environment={"YT_TOKEN": self.get_vault_data('VIDEODEV', 'yt_token')},
            outputs_to_one_file=False,
            log_prefix="yt.{}".format(cmd),
            timeout=600,
            wait=False,
            stdout=output
        )

    def _read_yt_file(self, file_path, dst_path):
        yt_exists = self._yt(None, "exists", file_path)
        if yt_exists.wait():
            raise errors.SandboxTaskFailureError("Failed to check existence %s" % (file_path))

        exists = False
        with open(yt_exists.stdout_path, 'r') as f:
            exists = str.strip(f.readline()) == "true"

        if not exists:
            return False

        with open(dst_path, "w") as output:
            yt_read_file = self._yt(output, "read-file", "--path", file_path)
            if yt_read_file.wait():
                raise errors.SandboxTaskFailureError("Failed to read {}".format(file_path))
        return True

    def _build_ban(self):
        local_path = self.abs_path("./videohub_supertags")
        if not self._read_yt_file(self.ctx[YtSuperTags.name], local_path):
            return 0

        has_changes = self._update_resource(VIDEO_SUPERTAGS, path=local_path)
        if not has_changes:
            return 0

        return os.stat(local_path).st_size

__Task__ = VideoReleaseSuperTags
