import os
import shutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.constants import constants
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.nanny import nanny

from . import packages as pkg
from . import resource_types  # noqa


class VideoContentIdBuildPackages(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Context(sdk2.Task.Context):
        child_tasks = {}

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = build_params.ArcadiaUrl()
        arcadia_patch = build_params.ArcadiaPatch()

        with sdk2.parameters.CheckGroup('Packages') as packages:
            for item in pkg.PACKAGES_LIST:
                key = item['key']
                packages.values[key] = key

    def _build_package(self, package):
        resource_type = package['resource_type']
        arcadia_path = package['arcadia_path']
        description = 'Child of {}. Build {}'.format(self.id, resource_type)
        resource_description = 'Svn url for arcadia: {}. With patch: {}'.format(
            self.Parameters.checkout_arcadia_from_url,
            bool(self.Parameters.arcadia_patch),
        )

        child_task = sdk2.Task['YA_PACKAGE'](
            self,
            description=description,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            arcadia_patch=self.Parameters.arcadia_patch,
            use_aapi_fuse=True,
            aapi_fallback=True,
            package_resource_description={arcadia_path: resource_description},
            build_system=constants.SEMI_DISTBUILD_BUILD_SYSTEM,
            packages=arcadia_path,
            package_type=YaPackage.TARBALL,
            raw_package=True,
            resource_type=resource_type,
        )

        child_task.save().enqueue()
        assert resource_type not in self.Context.child_tasks
        self.Context.child_tasks[resource_type] = child_task.id
        self.set_info('{} build task {} started'.format(resource_type, child_task.id))

    def _build(self):
        for key in self.Parameters.packages:
            self._build_package(pkg.PACKAGES_DICT[key])

        raise sdk2.WaitTask(self.Context.child_tasks.values(), ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _prepare_resources(self):
        for resource_type, task_id in self.Context.child_tasks.items():
            task_status = sdk2.Task[task_id].status
            if task_status != ctt.Status.SUCCESS:
                raise errors.TaskFailure('Child task {} finished with status {}'.format(task_id, task_status))

            resource = sdk2.Resource.find(task_id=task_id, type=resource_type).first()
            if not resource:
                raise errors.TaskFailure('Failed to find {} resource of {}'.format(resource_type, task_id))

            path = sdk2.ResourceData(resource).path.as_posix()
            new_path = resource_type.lower()

            if os.path.isdir(path):
                shutil.copytree(path, new_path)
            else:
                shutil.copy(path, new_path)

            new_resource = sdk2.Resource[resource_type](self, resource.description, new_path)
            sdk2.ResourceData(new_resource).ready()

        self.set_info('Done')

    def on_execute(self):
        with self.memoize_stage.build:
            self._build()

        with self.memoize_stage.prepare_resources:
            self._prepare_resources()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
