import shutil

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import errors

from sandbox.projects import resource_types
from sandbox.projects.video import video_resource_types

from sandbox.projects.common.balancer_sepe.run_tests import MediaTestBalancer, ConfigArchiveResourceParameter


class VIDEO_MMETA_GENERATOR_CONFIGS(video_resource_types.VIDEO_RELEASABLE_RESOURCE):
    file_name = 'generator_configs.mmeta.7z'


class VIDEO_INT_GENERATOR_CONFIGS(video_resource_types.VIDEO_RELEASABLE_RESOURCE):
    file_name = 'generator_configs.int.7z'


class ReleaseMmetaConfigs(parameters.SandboxBoolParameter):
    name = 'release_mmeta_configs'
    description = 'Release metasearch config'
    default_value = True
    resource_type = VIDEO_MMETA_GENERATOR_CONFIGS


class ReleaseIntConfigs(parameters.SandboxBoolParameter):
    name = 'release_int_configs'
    description = 'Release intmetasearch config'
    default_value = True
    resource_type = VIDEO_INT_GENERATOR_CONFIGS


class ReleaseBdiConfigs(parameters.SandboxBoolParameter):
    name = 'release_bdi_configs'
    description = 'Release bdi configs'
    default_value = True


def get_corresponding_7z_archive(generator_resource):
    """
        The problem:
        gencfg releases two similar resources:
        CONFIG_GENERATOR_GENERATED  -- tar.gz compressed configs
        and
        CONFIG_GENERATOR_CONFIGS -- 7z compressed configs

        MediaTestBalancer tasks uses CONFIG_GENERATOR_GENERATED resource, unpacks it
        Production services use CONFIG_GENERATOR_CONFIGS

        getting 7z into sandbox tasks (to refactor MediaTestBalancer) seemed too complicated and unstable.
        So I decided to use CONFIG_GENERATOR_GENERATED for bdi task but release corresponding CONFIG_GENERATOR_CONFIGS (from the same task)
    """
    gencfg_task = generator_resource.task_id
    resource_type = resource_types.CONFIG_GENERATOR_CONFIGS
    config_resource = channel.sandbox.list_resources(task_id=gencfg_task, resource_type=resource_type)
    if len(config_resource) != 1:
        raise errors.SandboxTaskFailureError("Expected single resource of type {} in task {}".format(resource_type, gencfg_task))
    return config_resource[0]


class VideoReleaseGeneratorConfigs(MediaTestBalancer):
    """
        Release video service static discovery configs
    """

    type = 'VIDEO_RELEASE_GENERATOR_CONFIGS'

    input_parameters = [
        ReleaseBdiConfigs,
        ReleaseMmetaConfigs,
        ReleaseIntConfigs
    ]

    resource_options = (ReleaseMmetaConfigs, ReleaseIntConfigs)

    def on_enqueue(self):
        if self.ctx[ReleaseBdiConfigs.name]:
            MediaTestBalancer.on_enqueue(self)

    def on_execute(self):
        configs_resource = channel.sandbox.get_resource(self.ctx[ConfigArchiveResourceParameter.name])
        config_attributes = {
            'tag': configs_resource.attributes['tag']
        }

        input_resource = self.sync_resource(get_corresponding_7z_archive(configs_resource))
        for option in self.resource_options:
            if self.ctx[option.name]:
                shutil.copyfile(input_resource, option.resource_type.file_name)
                self.create_resource(configs_resource.attributes['tag'], option.resource_type.file_name, option.resource_type, arch='any', attributes=config_attributes)

        if self.ctx[ReleaseBdiConfigs.name]:
            MediaTestBalancer.on_execute(self)


__Task__ = VideoReleaseGeneratorConfigs
