import logging
import re

from sandbox.common import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk.channel import channel

from sandbox.projects import resource_types
from sandbox.projects.common import decorators
from sandbox.projects.common import apihelpers
from sandbox.projects.common import utils
from sandbox.projects.common import dolbilka
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.priemka import BasePriemkaTask as priemka_task
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.search import components as search_components
from sandbox.projects.common.testenv_client import TEClient

from sandbox.projects import BuildVideosearchBundle as build_task
from sandbox.projects.GetVideoSearchDatabase import GetVideoSearchDatabase
from sandbox.projects.VideoLoadBasesearchResources.task import VideoLoadBasesearchResources

from . import compare_performance

_GENERATION_TASKS = "generation_tasks"  # Context key to store plan generation task ids
_PERFORMANCE_TASKS = "analyze_performance_tests"  # Context key to store performance task ids

_BUILD_TASK_TYPE = build_task.BuildVideosearchBundle.type
_VIDEOMAIN_GROUP = "Video main basesearch"


class BuildTaskId(parameters.TaskSelector):
    name = priemka_task.BasePriemkaTask.BUILD_TASK_KEY
    description = 'Build task id'
    task_type = 'BUILD_VIDEOSEARCH_BUNDLE'
    required = False


class TestVideosearchParameter(parameters.SandboxBoolParameter):
    name = 'test_videosearch'
    description = 'Test video basesearch'
    group = _VIDEOMAIN_GROUP
    default_value = True


class TestVideoquickParameter(parameters.SandboxBoolParameter):
    name = 'test_videoquick'
    description = 'Test videoquick'
    default_value = False


class BasesearchPlanParameter(parameters.ResourceSelector):
    name = 'basesearch_plan_resource_id'
    description = 'Basesearch Plan'
    group = _VIDEOMAIN_GROUP
    resource_type = resource_types.BASESEARCH_PLAN
    required = False


class BasesearchIndexParameter(parameters.ResourceSelector):
    name = 'basesearch_index_resource_id'
    description = 'Basesearch Index'
    group = _VIDEOMAIN_GROUP
    resource_type = resource_types.VIDEO_SEARCH_DATABASE
    required = False


class PlanParameter(parameters.ResourceSelector):
    name = 'dolbilo_plan_resource_id'
    description = 'Plan'
    group = dolbilka.DOLBILKA_GROUP
    resource_type = resource_types.VIDEO_MIDDLESEARCH_PLAN
    required = True


class AttributesParameter(parameters.SandboxStringParameter):
    name = 'attributes'
    description = 'Set additional attrs to resources (ex.: attr1=v1, attr2=v2)'
    do_not_copy = True


class BaseShardNameFormatParameter(parameters.SandboxStringParameter):
    name = 'base_shard_name_format'
    description = 'Format for basesearch shard name (accepts {prefix} and {state} templates)'
    default_value = '{prefix}-009-{state}'
    group = _VIDEOMAIN_GROUP


class VideoPriemkaBasesearchBinary(priemka_task.BasePriemkaTask,
                                   compare_performance.ComparePerformanceTask):
    """
        Creates release tag and built basesearch binary from this tag.
        Launch tests to compare performance with current production binary.
    """

    type = 'VIDEO_PRIEMKA_BASESEARCH_BINARY'

    ARCADIA_PROJECT = priemka_task.ArcadiaProject('video/base')
    GENCFG_INSTANCES_URL = 'https://api.gencfg.yandex-team.ru/{topology}/searcherlookup/groups/{gencfg_group}/instances'
    GENCFG_BASESEARCH_GROUP = 'VLA_VIDEO_PLATINUM_BASE'
    GENCFG_MMETASEARCH_GROUP = 'VLA_VIDEO_MMETA'
    BASESEARCH_PLAN_RESOURCE_TYPE = 'BASESEARCH_PLAN'

    input_parameters = (
        BuildTaskId,
        TestVideosearchParameter,
        BasesearchPlanParameter,
        BasesearchIndexParameter,
        BaseShardNameFormatParameter,
    ) + priemka_task.generate_priemka_parameters(_BUILD_TASK_TYPE)

    __HEADER_TEMPLATE = [
        "Test",
        "Task id",
        "Task status",
    ]

    __INDEX_STATE_PATTERN = re.compile(r'^export YT_STATE=(.*)$')

    def get_project(self):
        return self.ARCADIA_PROJECT

    @classmethod
    def get_build_task_type(cls):
        return _BUILD_TASK_TYPE

    # TODO: Use "build_all"
    def get_build_task_ctx(self):
        ctx = {}
        if utils.get_or_default(self.ctx, TestVideosearchParameter):
            ctx.update({
                "test_videosearch": True
            })

        return ctx

    def get_main_resource_type(self):
        return resource_types.VIDEOSEARCH_EXECUTABLE.name

    @property
    def footer(self):
        items = [{
            "<h4>Performance tests<h4>": self._performance_footer(_PERFORMANCE_TASKS),
        }]

        changelog_resource = self.get_plain_text_changelog_resource()
        if changelog_resource:
            changelog_template = "<a href='//proxy.sandbox.yandex-team.ru/{0}'>{0}</a>".format(changelog_resource)
        else:
            changelog_template = "<span style='color:red'>No changelog found, see logs for details</span>"

        items.append({
            "<h4>Other</h4>": {
                "header": [
                    {"key": "k1", "title": "&nbsp;"},
                    {"key": "k2", "title": "&nbsp;"},
                ],
                "body": {
                    "k1": ["Build task", "Changelog"],
                    "k2": ["<a href='/task/{0}/view'>{0}</a>".format(self.get_build_task_id()), changelog_template],
                },
            },
        })

        # items.append(self._get_ultra_footer())

        return [{"content": item} for item in items]

    def on_execute(self):
        priemka_task.BasePriemkaTask.on_execute(self)

        if _GENERATION_TASKS not in self.ctx:
            gen_tasks = {}
            for base_index_type in self.__get_index_types():
                gen_tasks[base_index_type] = self._generate_resources(media_settings.INDEX_MIDDLE, base_index_type)
            self.ctx[_GENERATION_TASKS] = gen_tasks

        if _PERFORMANCE_TASKS not in self.ctx:
            results = {}
            for index_type, supermind_mult, query_type, custom_ctx in self.__get_performance_tests():
                supermind_str = str(supermind_mult) if supermind_mult else 'none'
                key = ','.join((index_type, query_type, supermind_str))
                key += ',custom' if custom_ctx else ''
                results[key] = self._performance_test(index_type, query_type, supermind_mult, custom_ctx=custom_ctx)
            self.ctx[_PERFORMANCE_TASKS] = results

        self.sync_subtasks()
        self._validate_performance(_PERFORMANCE_TASKS)

    @decorators.memoize
    def _get_build_basesearch_executable(self):
        res = media_settings.VideoSettings.basesearch_executable_resource()
        res_type = str(res)
        if res_type in self.ctx:
            return self.ctx.get(res_type)
        return self._get_build_resource(res, sandboxapi.ARCH_LINUX)

    @decorators.memoize
    def _get_build_basesearch_config(self):
        res = media_settings.VideoSettings.basesearch_config_resource()
        res_type = str(res)
        if res_type in self.ctx:
            return self.ctx.get(res_type)
        return self._get_build_resource(res, sandboxapi.ARCH_ANY)

    def _subtask_description(self, index_type, binary=None, supermind_mult=None, query_type=None):
        description = "{}, {}".format(self.get_description(), index_type)

        if supermind_mult is not None:
            description += ', mult={}'.format(supermind_mult)

        if query_type is not None:
            description += ', queries={}'.format(query_type)

        if binary is not None:
            description += ',{}'.format(binary)

        return description

    def _subtask_compare_ctx(self, index_type, basesearch_params):
        sub_ctx = {}
        for params, age in zip(basesearch_params, ("old", "new")):
            sub_ctx.update(self._subtask_basesearch_ctx(index_type, params, age))
        return sub_ctx

    def _subtask_basesearch_ctx(self, index_type, basesearch_params, age="new"):
        if age == "new":
            basesearch_executable = self._get_build_basesearch_executable()
            basesearch_config = self._get_build_basesearch_config()
        elif age == "old":
            basesearch_executable = self._get_basesearch_executable()
            basesearch_config = self._get_basesearch_config(index_type)

        basesearch_models = self._get_basesearch_models()

        sub_ctx = {
            basesearch_params.Binary.name: basesearch_executable,
            basesearch_params.Config.name: basesearch_config,
        }

        if basesearch_models:
            sub_ctx[basesearch_params.ArchiveModel.name] = basesearch_models

        _update_if_exists(sub_ctx, basesearch_params, "PoliteMode", False)
        _update_if_exists(sub_ctx, basesearch_params, "PatchRequestThreads", False)

        return sub_ctx

    @decorators.memoize
    def _subtask_plan_id(self, index_type, query_type):
        plan_id = self.ctx.get(BasesearchPlanParameter.name, None)
        if plan_id:
            return plan_id

        generation_task_id = self.ctx[_GENERATION_TASKS][index_type][VideoLoadBasesearchResources.type]
        if not generation_task_id:
            raise errors.TaskFailure(
                'No {} task found for index_type "{}"'.format(VideoLoadBasesearchResources.type, index_type))

        plans = apihelpers.list_task_resources(generation_task_id, resource_types.BASESEARCH_PLAN, limit=1000)

        for plan in plans:
            if query_type in plan.attributes:
                logging.info('Using plan {} for index_type "{}" and query_type "{}" from task {}'
                             .format(plan.id, index_type, query_type, generation_task_id))
                return plan.id

        raise errors.TaskFailure(
            'No plan found for index_type "{}" and query_type "{}" in task {}, total plans: {}'
            .format(index_type, query_type, generation_task_id, len(plans)))

    @decorators.memoize
    def _subtask_database_id(self, index_type):
        database_id = self.ctx.get(BasesearchIndexParameter.name, None)
        if database_id:
            return database_id

        generation_task_id = self.ctx[_GENERATION_TASKS][index_type][GetVideoSearchDatabase.type]
        if not generation_task_id:
            raise errors.TaskFailure(
                'No {} task found for {}'.format(GetVideoSearchDatabase.type, index_type))

        databases = apihelpers.list_task_resources(generation_task_id, resource_types.VIDEO_SEARCH_DATABASE,
                                                   limit=1000)
        if databases and len(databases) > 0:
            logging.info('Using database {} for index_type {} from task {}'
                         .format(databases[0].id, index_type, generation_task_id))
            return databases[0].id

        raise errors.TaskFailure(
            'No database found for index_type {} in task {}, total database resources: {}'
            .format(index_type, generation_task_id, len(databases)))

    @decorators.memoize
    def _subtask_meta_queries_id(self, meta_type, index_type):
        testenv_attributes = media_settings.VideoSettings.testenv_middlesearch_queries_attributes(
            media_settings.VideoSettings.MMETA_ID,
            media_settings.VideoSettings.PRIEMKA_SOURCE[1]
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
            testenv_attributes[0],
            testenv_attributes[1],
        ).id

    def _get_basesearch_attributes(self, index_type, serp_type):
        return ("priemka", "yes")

    def __get_index_types(self):
        return set(test[0] for test in self.__get_performance_tests())

    def __get_performance_tests(self):
        results = []
        if utils.get_or_default(self.ctx, TestVideosearchParameter):
            results.extend([
                (media_settings.INDEX_MAIN, None, 'video_serp_search', {}),
                (media_settings.INDEX_MAIN, None, 'video_serp_factors', {}),
                (media_settings.INDEX_MAIN, None, 'video_snippets', {'threshold': {'shooting.rps_0.5': -3.0}}),
                (media_settings.INDEX_MAIN, None, 'video_related_search', {}),
                (media_settings.INDEX_MAIN, None, 'video_related_factors', {}),
                (media_settings.INDEX_MAIN, None, 'video_testenv_base_main_vla_videohub_search', {}),
            ])
        if utils.get_or_default(self.ctx, TestVideoquickParameter):
            results.append((media_settings.INDEX_QUICK, None, 'video_serp_search', {'threshold': {'shooting.rps_0.5': -1.5}}))
            results.append((media_settings.INDEX_QUICK, None, 'video_serp_factors', {'threshold': {'shooting.rps_0.5': -1.5}}))
            results.append((media_settings.INDEX_QUICK, None, 'video_snippets', {'threshold': {'shooting.rps_0.5': -3.0}}))

        return results

    def _get_basesearch_executable(self):
        service_id = media_settings.VideoSettings.SBASE_NANNY_SERVICE_NAME
        logging.info('Load basesearch executable from nanny service: {}'.format(service_id))
        return self.__get_nanny_file(
            service_id,
            media_settings.VideoSettings.basesearch_executable_resource()
        )

    @decorators.memoize
    def __get_nanny_file(self, service_id, resource_type):
        task_id = self.__get_nanny_task(service_id, resource_type)
        return self.__get_sandbox_resource(task_id, resource_type).id

    def __get_subtask(self, testenv_data_task_id, task_type):
        load_tasks = channel.sandbox.list_tasks(task_type=task_type,
                                                     parent_id=testenv_data_task_id)
        if load_tasks and len(load_tasks) > 0:
            task_id = load_tasks[0].id
            logging.info('Found {} {} tasks for parent {}. Using task {}'
                         .format(len(load_tasks), task_type, testenv_data_task_id, task_id))
            return task_id

        logging.info('Found no {} tasks with plans for parent {}'.format(task_type, testenv_data_task_id))
        return None

    def _generate_resources(self, meta_index_type, base_index_type,
                            serp_type=None):
        """
            Generate plans for basesearch
        """
        testenv_data_task = TEClient.get_last_sandbox_task_ids('production_resources', ['GET_VIDEO_PLANS_AND_REQS'],
                                                               success=True)
        testenv_data_task = testenv_data_task[0]['task_id']

        result = {
            VideoLoadBasesearchResources.type: self.__get_subtask(testenv_data_task, VideoLoadBasesearchResources.type),
            GetVideoSearchDatabase.type: self.__get_subtask(testenv_data_task, GetVideoSearchDatabase.type)
        }

        return result

    def _get_basesearch_models(self):
        return self.__get_nanny_file(
            media_settings.VideoSettings.SBASE_NANNY_SERVICE_NAME,
            media_settings.VideoSettings.DEFAULT_BASESEARCH_MODELS_RES_TYPE
        )

    @decorators.memoize
    def _get_middlesearch_executable(self):
        return self.__get_nanny_file(
            media_settings.VideoSettings.SMIDDLE_NANNY_SERVICE_NAME,
            resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE
        )

    @decorators.memoize
    def _get_middlesearch_config(self):
        attribute_name, attribute_value = media_settings.VideoSettings.testenv_resource_attributes(
            media_settings.VideoSettings.MMETA_ID,
            media_settings.VideoSettings.PRIEMKA_SOURCE[1])

        return utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_CONFIG,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_data(self):
        attribute_name, attribute_value = media_settings.VideoSettings.testenv_resource_attributes(
            media_settings.VideoSettings.MMETA_ID,
            media_settings.VideoSettings.PRIEMKA_SOURCE[1]
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_DATA,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_plan(self, meta_index_type, base_index_type, serp_type=None):
        attribute_name, attribute_value = media_settings.VideoSettings.testenv_middlesearch_queries_attributes(
            media_settings.VideoSettings.MMETA_ID,
            media_settings.VideoSettings.PRIEMKA_SOURCE[1]
        )
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_PLAN,
            attribute_name,
            attribute_value
        ).id

    @decorators.memoize
    def _get_middlesearch_models(self):
        return self.__get_nanny_file(
            media_settings.VideoSettings.SMIDDLE_NANNY_SERVICE_NAME,
            resource_types.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE
        )

    @decorators.memoize
    def _get_middlesearch_index(self):
        shard_name = self.__get_middle_shard_name()
        return utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_INDEX,
            media_settings.SHARD_INSTANCE_ATTRIBUTE_NAME,
            shard_name
        ).id

    def create_middlesearch_params(self):
        return search_components.create_middlesearch_params(
            component_name="middlesearch",
            group_name="Middlesearch",
            use_int=False
        )

    def create_basesearch_params(self, component_name=None):
        return search_components.create_basesearch_params(
            component_name="basesearch" if not component_name else component_name,
            group_name="Basesearch",
        )

    def _get_basesearch_config(self, index_type=media_settings.INDEX_MAIN):
        config_resource = media_settings.VideoSettings.basesearch_config_resource()
        try:
            # use fake config for test if exists
            return self.__get_nanny_file(
                media_settings.VideoSettings.SBASE_NANNY_SERVICE_NAME,
                config_resource
            )
        except errors.TaskFailure as e:
            logging.info(e)
            return self.__get_sandbox_resource(
                self._get_basesearch_task(index_type),
                config_resource
            ).id

    def _get_basesearch_task(self):
        return self.__get_nanny_task(
            media_settings.VideoSettings.SBASE_NANNY_SERVICE_NAME,
            media_settings.VideoSettings.basesearch_executable_resource()
        )

    @decorators.memoize
    def __get_nanny_task(self, service_id, resource_type):
        sandbox_resources = self.__get_nanny_resource(service_id)['sandbox_files']
        resource_type = str(resource_type)

        for resource in sandbox_resources:
            if resource['resource_type'] == resource_type:
                return resource['task_id']

        raise errors.TaskFailure('Failed to find sandbox resource of type {}'.format(resource_type))

    @decorators.memoize
    def __get_sandbox_resource(self, task_id, resource_type):
        task_resources = apihelpers.list_task_resources(task_id, resource_type=resource_type)
        if not task_resources:
            raise errors.TaskFailure('Failed to find resource {} in task {}'.format(resource_type, task_id))
        return task_resources[0]

    @decorators.retries(max_tries=3, delay=10)
    def __get_nanny_resource(self, service_id):
        nanny_client = self.__get_nanny_client()

        # search for active snapshot id
        # TODO: simplify after SWAT-2375
        current_state = nanny_client.get_service_current_state(service_id)
        if current_state['content']['summary']['value'] not in ('ONLINE', 'UPDATING', 'PREPARING'):
            raise errors.TaskFailure('Service {} is offline'.format(service_id))
        for snapshot in current_state['content']['active_snapshots']:
            if snapshot['state'] in ('ACTIVATING', 'GENERATING', 'PREPARING'):
                active_snapshot_id = current_state['content']['rollback_snapshot']['snapshot_id']
                break
            elif snapshot['state'] in ('ACTIVE'):
                active_snapshot_id = snapshot['snapshot_id']
                break
        else:
            raise errors.TaskFailure('Failed to find active snapshot')

        # retrieve information about shardmap
        runtime_attrs = nanny_client.get_history_runtime_attrs(active_snapshot_id)
        return runtime_attrs['content']['resources']

    def __get_nanny_shard_map(self, service_id):
        shardmap_data = nanny.get_nanny_runtime_shardmap_info(service_id,
                                                              media_settings.VideoSettings.get_nanny_token(self))
        shardmap_resource_id = self.__get_sandbox_resource(shardmap_data['task_id'], shardmap_data['resource_type']).id
        return self.sync_resource(shardmap_resource_id)

    def __get_nanny_client(self):
        return nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('VIDEO-ROBOT', 'robot-video-crawl-nanny-oauth')
        )

    def __get_base_shard_name(self):
        service_id = media_settings.VideoSettings.SBASE_NANNY_SERVICE_NAME
        nanny_client = self.__get_nanny_client()

        resources = nanny_client.get_service_resources(service_id)
        files = resources['content']['sandbox_files']
        index_state_resource = None
        for f in files:
            if f['resource_type'] == 'VIDEO_PRODUCTION_INDEX_STATE_RESOURCE':
                index_state_resource = f['resource_id']
                break

        if not index_state_resource:
            raise errors.TaskFailure('Failed to find VIDEO_PRODUCTION_INDEX_STATE_RESOURCE for {}'.format(service_id))

        index_state = self.sync_resource(index_state_resource)
        state = None
        with open(index_state) as index_state_file:
            for line in index_state_file:
                m = VideoPriemkaBasesearchBinary.__INDEX_STATE_PATTERN.match(line.strip())
                if m:
                    state = m.group(1)

        if not state:
            raise errors.TaskFailure('Failed to find {} in sandbox resource {}'
                                                 .format(VideoPriemkaBasesearchBinary.__INDEX_STATE_PATTERN.pattern,
                                                         index_state_resource))

        shard_name = utils.get_or_default(self.ctx, BaseShardNameFormatParameter).format(
            prefix=media_settings.VideoSettings.DATABASE_PREFIXES[media_settings.INDEX_MAIN],
            state=state
        )
        logging.info('Base shard name = {}'.format(shard_name))

        return shard_name

    def __get_middle_shard_name(self):
        shard_prefix = media_settings.VideoSettings.DATABASE_PREFIXES[media_settings.INDEX_MIDDLE]
        shard_prefix = "{}-".format(shard_prefix)
        shardmap_path = self.__get_nanny_shard_map(media_settings.VideoSettings.SMIDDLE_NANNY_SERVICE_NAME)

        with open(shardmap_path) as shardmap_file:
            for line in shardmap_file:
                line = line.strip().split()
                if len(line) < 2 or not line[1].startswith(shard_prefix):
                    continue
                shard_name = line[1]
                shard_attrs = shard_name.find("(")
                if shard_attrs > 0:
                    shard_name = shard_name[:shard_attrs]
                return shard_name
            else:
                raise errors.TaskFailure("Failed to read shardmap name from shardmap")


def _update_if_exists(sub_ctx, params, attr, value):
    """Update dictionary if specified attribute exists in parameters"""

    if hasattr(params, attr):
        sub_ctx[getattr(params, attr).name] = value


__Task__ = VideoPriemkaBasesearchBinary
