# -*- coding: utf-8 -*-
from sandbox import sdk2
import time
from sandbox import common
import sandbox.common.types.task as ctt
from sandbox.projects.video.priemka.launchers.mlm_launcher import MlmTaskLauncher
from sandbox.projects.video.priemka.launchers.performance_launcher import PerformanceTaskLauncher
from sandbox.projects.video.priemka.launchers.sharp_eye_launcher import SharpEyeTaskLauncher
from sandbox.common import rest
from sandbox.common.proxy import OAuth
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.common.sdk_compat import task_helper as th
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import metrics_launch_manager as metrics_launcher


class VideoRobotPriemka(sdk2.Task):
    """
        Video robot priemka task

        Launch flow:
        Check previous launch and compare shardmap id's - if same, than no launch needed
        Check if given acceptance service switched to new shardmap (compare target and current nanny state)
        Prepare mlm params (simple)
        Prepare performance params:
            - Check if resources should be uploaded to sandbox (if task is launched on local)
            - Prepare task params
        Launch subtasks and raise for status

        Subtasks will be restarted automatically if failed. Max restarts provided from input
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Video robot acceptance"
        max_restarts = 0
        kill_timeout = 20 * 3600

        environment = sdk2.parameters.String("Environment", default="PRODUCTION", choices=[(x, x) for x in ["PRODUCTION", "BETA", "CUSTOM"]])

        with environment.value["CUSTOM"]:
            nanny_service_name = sdk2.parameters.String("Nanny acceptance service name", default='pip_video_base')
            max_wait = sdk2.parameters.Integer("How long to wait for service to switch (in seconds)", default=20 * 3600)
            baseline_host = sdk2.parameters.String("Baseline host", default="hamster")
            acceptance_host = sdk2.parameters.String("Acceptance host", default="newdb-vid-pip.hamster")
            telegram_block = sdk2.parameters.Info("Telegram settings")
            telegram_notification = sdk2.parameters.Bool("Telegram notifications", default=False)

            with telegram_notification.value[True]:
                telegram_chat_id = sdk2.parameters.String("Telegram chat id", default="-1001089802374")

            mlm_block = sdk2.parameters.Info("MLM Parameters")
            test_mlm = sdk2.parameters.Bool("Run MLM acceptance", default=True)

            with test_mlm.value[True]:
                mlm_max_attempts = sdk2.parameters.Integer("Max tries", default=3)
                baseline_profile = sdk2.parameters.String("Baseline profile", default="hamster")
                acceptance_profile = sdk2.parameters.String("Acceptance profile", default="hamster")

            perf_block = sdk2.parameters.Info("Performance Parameters")
            test_performance = sdk2.parameters.Bool("Run Performance acceptance", default=True)

            with test_performance.value[True]:
                performance_max_attempts = sdk2.parameters.Integer("Max tries", default=3)
                beta_service = sdk2.parameters.String("Beta service", default="NANNY", choices=[(x, x) for x in ["NANNY"]])

            sharp_eye_block = sdk2.parameters.Info("SharpEye Parameters")
            test_sharp_eye = sdk2.parameters.Bool("Run SharpEye acceptance", default=True)

            with test_sharp_eye.value[True]:
                sharp_eye_max_attempts = sdk2.parameters.Integer("Max tries", default=3)

            with sdk2.parameters.Group("OAuth parameters") as oauth_params:
                sb_rest_api_server = sdk2.parameters.String("Sandbox REST API server", default="https://sandbox.yandex-team.ru:443/api/v1.0")
                sb_oauth_item = sdk2.parameters.String("Sandbox OAuth vault item", default='sandbox-oauth-token')
                sb_oauth_owner = sdk2.parameters.String("Sandbox OAuth vault owner", default='VIDEO-BASE-DEPLOY')
                nanny_oauth_item = sdk2.parameters.String("Nanny OAuth vault item", default='nanny-oauth-token')
                nanny_oauth_owner = sdk2.parameters.String("Nanny OAuth vault owner", default='VIDEO-BASE-DEPLOY')

            with test_mlm.value[True]:
                mlm_oauth_item = sdk2.parameters.String("OAuth vault item", default='oauth_token')
                mlm_oauth_owner = sdk2.parameters.String("OAuth vault owner", default='VIDEO-BASE-DEPLOY')

    class Context(sdk2.Task.Context):
        shardmap_name = None
        mlm_launched = False
        mlm_num_launches = 0
        performance_launched = False
        performance_num_launches = 0
        sharp_eye_num_launches = 0
        shardmap_task_id = None
        need_launch = False

    def get_runtime_shardmap_name(self):
        self.set_info("Get shardmap name from nanny service {}".format(self.Parameters.nanny_service_name))
        shardmap_info = nanny.get_nanny_runtime_shardmap_info(
            self.Parameters.nanny_service_name,
            th.vault_data(
                self.Parameters.nanny_oauth_owner,
                self.Parameters.nanny_oauth_item,
            ),
        )
        shardmap_task_id = int(shardmap_info['task_id'])
        shardmap_resource_type = shardmap_info['resource_type']
        res = rest.Client(
            base_url=str(self.Parameters.sb_rest_api_server),
            auth=OAuth(
                th.vault_data(
                    self.Parameters.sb_oauth_owner,
                    self.Parameters.sb_oauth_item,
                )
            )
        ).resource.read(
            type=shardmap_resource_type,
            task_id=shardmap_task_id,
            limit=1
        )
        shardmap = res['items'][0]['attributes']['shardmap_name']
        self.set_info("Shardmap: {}".format(shardmap))
        return shardmap

    def create_mlm_task(self):
        self.set_info('Launching mlm')
        mlm_launcher = MlmTaskLauncher(self)
        mlm_launcher.set_launch_params(
            env=self.Parameters.environment,
            owner=self.owner
        )
        mlm_launcher.set_vault_params(
            item=self.Parameters.mlm_oauth_item,
            owner=self.Parameters.mlm_oauth_owner
        )
        mlm_launcher.set_hosts(
            baseline_host=self.baseline_host,
            baseline_profile=self.baseline_profile,
            acceptance_host=self.acceptance_host,
            acceptance_profile=self.acceptance_profile
        )
        task_info, subtask = mlm_launcher.launch_task()
        self.Context.mlm_launched = True
        self.Context.mlm_task_id = task_info['id']

        return subtask

    def create_performance_task(self):
        self.set_info('Launching performance')
        performance_launcher = PerformanceTaskLauncher(self)
        performance_launcher.set_sb_oauth(
            self.Parameters.sb_rest_api_server,
            self.Parameters.sb_oauth_owner,
            self.Parameters.sb_oauth_item
        )
        performance_launcher.set_nanny_service(
            self.Parameters.nanny_service_name
        )
        performance_launcher.set_nanny_oauth(
            self.Parameters.nanny_oauth_owner,
            self.Parameters.nanny_oauth_item
        )
        performance_launcher.set_hosts(
            self.baseline_host,
            self.acceptance_host
        )
        task_info, subtask = performance_launcher.launch_task()
        self.Context.performance_launched = True
        self.Context.performance_task_id = task_info['id']

        return subtask

    def create_sharp_eye_task(self):
        self.set_info('Launching SharpEye')
        sharp_eye_launcher = SharpEyeTaskLauncher(self)
        sharp_eye_launcher.set_launch_params(
            env=self.Parameters.environment
        )
        task_info, subtask = sharp_eye_launcher.launch_task()
        self.Context.sharp_eye_launched = True
        self.Context.sharp_eye_task_id = task_info['id']

        return subtask

    def wait_for_subtasks(self, subtasks):
        raise sdk2.WaitTask(
            subtasks,
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            wait_all=True
        )

    def launch_subtasks(self):
        child_tasks = []
        if self.need_performace():
            child_tasks.append(self.create_performance_task())
            self.Context.performance_num_launches += 1
        if self.need_mlm():
            child_tasks.append(self.create_mlm_task())
            self.Context.mlm_num_launches += 1
        if self.need_sharp_eye():
            child_tasks.append(self.create_sharp_eye_task())
            self.Context.sharp_eye_num_launches += 1
        self.wait_for_subtasks(child_tasks)

    def _get_task(self, task_id):
        return sdk2.Task.find(
            id=task_id,
            children=True
        ).limit(1).first()

    def need_mlm(self):
        if not self.test_mlm:
            return False
        if self.Context.mlm_launched:
            return False
        return True

    def need_sharp_eye(self):
        if not self.test_sharp_eye:
            return False
        if self.Context.sharp_eye_launched:
            return False
        return True

    def need_performace(self):
        if not self.test_performance:
            return False
        if self.Context.performance_launched:
            return False
        return True

    def need_launch(self):
        if self.test_mlm:
            if not self.Context.mlm_launched:
                return True
            if self._get_task(self.Context.mlm_task_id).status != ctt.Status.SUCCESS:
                self.Context.mlm_launched = False
                if self.Context.mlm_num_launches >= self.Parameters.mlm_max_attempts:
                    self.fail_telegram_notification()
                    raise common.errors.TaskError("MLM task failed after {} attemts".format(self.Context.mlm_num_launches))
        if self.test_performance:
            if not self.Context.performance_launched:
                return True
            if self._get_task(self.Context.performance_task_id).status != ctt.Status.SUCCESS:
                self.Context.performance_launched = False
                if self.Context.performance_num_launches >= self.Parameters.performance_max_attempts:
                    self.fail_telegram_notification()
                    raise common.errors.TaskError("Performance task failed after {} attemts".format(self.Context.performance_num_launches))
        if self.test_sharp_eye:
            if not self.Context.sharp_eye_launched:
                return True
            if self._get_task(self.Context.sharp_eye_task_id).status != ctt.Status.SUCCESS:
                self.Context.sharp_eye_launched = False
                if self.Context.sharp_eye_num_launches >= self.Parameters.sharp_eye_max_attempts:
                    self.fail_telegram_notification()
                    raise common.errors.TaskError("SharpEye task failed after {} attemts".format(self.Context.sharp_eye_num_launches))

        return self.need_mlm() or self.need_performace() or self.need_sharp_eye()

    def get_nanny_active_shardmap(self, nanny_server):
        active_state = nanny_server.get_service_active_runtime_attrs(self.Parameters.nanny_service_name)
        return active_state['content']['resources']['sandbox_bsc_shard']['sandbox_shardmap']['task_id']

    def get_nanny_target_shardmap(self, nanny_server):
        current_state = nanny_server.get_service_description(self.Parameters.nanny_service_name)
        return current_state['runtime_attrs']['content']['resources']['sandbox_bsc_shard']['sandbox_shardmap']['task_id']

    def wait_nanny_to_switch(self):
        nanny_server = NannyClient(
            rm_const.Urls.NANNY_BASE_URL,
            th.vault_data(
                self.Parameters.nanny_oauth_owner,
                self.Parameters.nanny_oauth_item,
            )
        )

        self.set_info("Wait for service {} to switch to new shardmap".format(self.Parameters.nanny_service_name))
        target_id = self.get_nanny_target_shardmap(nanny_server)
        waiting_time = 0
        __SLEEP_TIME = 10 * 60  # 10 minutes
        while waiting_time <= self.Parameters.max_wait:
            active_id = self.get_nanny_active_shardmap(nanny_server)
            if active_id == target_id:
                break
            self.set_info(
                "Target shardmap task_id: {}\n".format(target_id) +
                "Active shardmap task_id: {}\n".format(active_id) +
                "Wait for {} seconds".format(__SLEEP_TIME)
            )
            time.sleep(__SLEEP_TIME)
            waiting_time += __SLEEP_TIME

        if active_id == target_id:
            self.set_info("Service {} switched successfully".format(self.Parameters.nanny_service_name))
            return

        raise common.errors.TaskError("Service {} updating too long.\n{}\nCheck service and restart".format(
            self.Parameters.nanny_service_name,
            'https://nanny.yandex-team.ru/ui/#/services/catalog/' + self.Parameters.nanny_service_name
            )
        )

    def generate_report(self):
        telegram_message = self.generate_finish_message()
        self.send_telegram_notification(telegram_message)

    def get_mlm_status(self, launch_id):
        regions = ["RU", "TR"]
        all_borders = self.tasks_borders["mlm"]

        launcher = metrics_launcher.MetricsLauncher(
            oauth_token=sdk2.Vault.data(self.Parameters.mlm_oauth_owner, self.Parameters.mlm_oauth_item)
        )

        launches = launcher.get_launch_info(launch_id)["launches"]
        status = "GOOD"

        for launch in launches:
            if launch["regionalType"] in regions:

                for query in launch["diffQueryGroups"]:
                    borders = all_borders["porno"] if "porno" in query["preFilterNames"] else all_borders["default"]

                    for metric in query["metrics"]:
                        if metric["metricId"] in borders:
                            metric_border = borders[metric["metricId"]]
                            diff_percent = metric["diffPercent"]

                            if diff_percent < metric_border["WARN"]["min_diff_percent"] or diff_percent > metric_border["WARN"]["max_diff_percent"]:
                                if status != "CRIT":
                                    status = "WARN"
                            if diff_percent < metric_border["CRIT"]["min_diff_percent"] or diff_percent > metric_border["CRIT"]["max_diff_percent"]:
                                status = "CRIT"

        return status

    def get_performance_status(self):
        results = sdk2.Task.find(
            id=self.Context.performance_task_id,
            children=True,
            status=ctt.Status.SUCCESS
        ).first().Context.analyze_results

        max_percent = 0
        for query in results:
            tests = query[2]

            for test in tests:
                max_percent = min(max_percent, test["median"])

        status = "GOOD"
        borders = self.tasks_borders["performance"]

        if max_percent < borders["WARN"]:
            status = "WARN"
        if max_percent < borders["CRIT"]:
            status = "CRIT"

        return status

    def init_telegram_bot(self):
        telegram_token_name = "video-warden-bot-token"
        token_owner = "VIDEO-ROBOT"
        self.telegram_chat_id = self.Parameters.telegram_chat_id
        bot_token = th.vault_data(token_owner, telegram_token_name)
        self.telegram_bot = common.telegram.TelegramBot(bot_token)

    def start_telegram_notification(self):
        message = "Video robot acceptance ({}) started. Sandbox task https://sandbox.yandex-team.ru/task/{}".format(
            self.Parameters.environment,
            self.id
        )
        self.send_telegram_notification(message)

    def fail_telegram_notification(self):
        message = "*FAIL* Video robot acceptance ({}). Sandbox task https://sandbox.yandex-team.ru/task/{}".format(
            self.Parameters.environment,
            self.id
        )
        self.send_telegram_notification(message)

    def mlm_telegram_notification(self):
        task = sdk2.Task.find(
            id=self.Context.mlm_task_id,
            children=True,
            status=ctt.Status.SUCCESS
        ).first()
        launch_id = task.Context.launch_info["launch_id"]

        status = self.get_mlm_status(launch_id)
        message = "\nMlm launch https://mlm.yandex-team.ru/sets/{} ({})".format(launch_id, status)

        return message

    def sharp_eye_telegram_notification(self):
        task = sdk2.Task.find(
            id=self.Context.sharp_eye_task_id,
            children=True,
            status=ctt.Status.SUCCESS
        ).first()
        baseline_pool = task.Context.pools["baseline"]
        acceptance_pool = task.Context.pools["acceptance"]

        message = "\nSharp eye https://fml.yandex-team.ru/view/factor/diff?right-pool-id={acceptance}&left-pool-id={baseline}"\
                                .format(baseline=baseline_pool, acceptance=acceptance_pool)

        return message

    def performance_telegram_notification(self):
        status = self.get_performance_status()
        message = "\nPerformance sandbox task https://sandbox.yandex-team.ru/task/{} ({})".format(self.Context.performance_task_id, status)

        return message

    def send_telegram_notification(self, message):
        try:
            self.telegram_bot.send_message(self.telegram_chat_id, message, parse_mode=common.telegram.ParseMode.MARKDOWN)
        except Exception as exc:
            self.set_info("Failed send telegram message " + message)
            self.set_info(str(exc))

    def generate_finish_message(self):
        mlm_report = self.mlm_telegram_notification() if self.test_mlm else ""
        performance_report = self.performance_telegram_notification() if self.test_performance else ""
        sharp_eye_report = self.sharp_eye_telegram_notification() if self.test_sharp_eye else ""

        return "*SUCCESS* Video robot acceptance ({}).".format(self.Parameters.environment) + mlm_report + performance_report + sharp_eye_report

    def init_params(self):
        environment_params = {
            "PRODUCTION": {
                "baseline_host": "hamster",
                "baseline_profile": "hamster",
                "acceptance_host": "newdb-vid-pip.hamster",
                "acceptance_profile": "hamster",
                "telegram_notification": True,
                "test_mlm": True,
                "test_performance": True,
                "test_sharp_eye": True,
                "beta_service": "NANNY"
            },
            "CUSTOM": {
                "baseline_host": self.Parameters.baseline_host,
                "baseline_profile": self.Parameters.baseline_profile,
                "acceptance_host": self.Parameters.acceptance_host,
                "acceptance_profile": self.Parameters.acceptance_profile,
                "telegram_notification": self.Parameters.telegram_notification,
                "test_mlm": self.Parameters.test_mlm,
                "test_performance": self.Parameters.test_performance,
                "test_sharp_eye": self.Parameters.test_sharp_eye,
                "beta_service": self.Parameters.beta_service
            }
        }

        self.tasks_borders = {
            "performance": {
                "CRIT": -5,
                "WARN": -3
            },
            "mlm": {
                "default": {
                    "video-player-pfound-5": {
                        "CRIT": {
                            "min_diff_percent": -1,
                            "max_diff_percent": 5
                        },
                        "WARN": {
                            "min_diff_percent": -0.5,
                            "max_diff_percent": 5
                        }
                    }
                },
                "porno": {
                    "video-player-pfound-5": {
                        "CRIT": {
                            "min_diff_percent": -2,
                            "max_diff_percent": 10
                        },
                        "WARN": {
                            "min_diff_percent": -1,
                            "max_diff_percent": 10
                        }
                    }
                }
            }
        }

        env_params = environment_params[self.Parameters.environment]

        self.baseline_host = env_params["baseline_host"]
        self.baseline_profile = env_params["baseline_profile"]
        self.acceptance_host = env_params["acceptance_host"]
        self.acceptance_profile = env_params["acceptance_profile"]
        self.telegram_notification = env_params["telegram_notification"]
        self.test_mlm = env_params["test_mlm"]
        self.test_performance = env_params["test_performance"]
        self.test_sharp_eye = env_params["test_sharp_eye"]
        self.beta_service = env_params["beta_service"]
        self.wait_switching_nanny = self.test_performance and self.beta_service == "NANNY"

        self.init_telegram_bot()

    def on_execute(self):
        self.init_params()

        with self.memoize_stage.set_shardmap_name:
            self.Context.shardmap_name = self.get_runtime_shardmap_name()

        with self.memoize_stage.wait_nanny_to_switch:
            if self.wait_switching_nanny:
                self.wait_nanny_to_switch()

        if self.need_launch():
            with self.memoize_stage.telegram_notification:
                if self.telegram_notification:
                    self.start_telegram_notification()

            self.launch_subtasks()

        if self.telegram_notification:
            self.generate_report()
