# -*- coding: utf-8 -*-
import os
import re
import time

from sandbox import sdk2
from sandbox import common
from sandbox.common.types.client import Tag
from sandbox.projects.common import metrics_launch_manager as metrics_launcher
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import error_handlers as eh


class VideoRobotPriemkaMlmReport(sdk2.Resource):
    releasable = False
    mlm_id = sdk2.parameters.String("MLM id")
    launch_name = sdk2.parameters.String("Launch name")
    laundh_id = sdk2.parameters.String("Launch id")


class VideoRobotPriemkaLaunchMlm(sdk2.Task):
    """
        Task to launch video robot mlm acceptance

        Launch mlm using metrics_launcher
        Wait finished
        Collect and save html report
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024  # 10 gb
        client_tags = Tag.GENERIC

    class Parameters(sdk2.Task.Parameters):
        description = "Video robot priemka mlm launcher"
        max_restarts = 1  # default is 10
        kill_timeout = 10 * 3600  # 6 hours
        environment = sdk2.parameters.String("Environment", default="PRODUCTION", choices=[(x, x) for x in ["PRODUCTION", "BETA", "CUSTOM"]])

        with environment.value["CUSTOM"]:
            with sdk2.parameters.Group("MLM parameters") as mlm_params:
                baseline_host = sdk2.parameters.String("Baseline host", default="hamster")
                baseline_profile = sdk2.parameters.String("Baseline profile", default="hamster")
                acceptance_host = sdk2.parameters.String("Acceptance host", default="newdb-vid-pip.hamster")
                acceptance_profile = sdk2.parameters.String("Acceptance profile", default="hamster")
                quota = sdk2.parameters.String("Scraper quota", default="video-robot")
                ang_quota = sdk2.parameters.String("AngQuota", default="VIDEO High Priority")

            with sdk2.parameters.Group("Vault parameters") as vault_params:
                vault_item = sdk2.parameters.String("OAuth vault item", default='oauth_token')
                vault_owner = sdk2.parameters.String("OAuth vault owner", default='VIDEO-BASE-DEPLOY')

    class Context(sdk2.Task.Context):
        launch_info = {}

    def init_params(self):
        environment_params = {
            "PRODUCTION": {
                "baseline_host": "hamster",
                "baseline_profile": "hamster",
                "acceptance_host": "newdb-vid-pip.hamster",
                "acceptance_profile": "hamster",
                "quota": "video-robot",
                "ang_quota": "VIDEO Acceptance"
            },
            "BETA": {
                "baseline_host": "priemka-vid.hamster",
                "baseline_profile": "c1",
                "acceptance_host": "betaindex.hamster",
                "acceptance_profile": "c1",
                "quota": "video-robot",
                "ang_quota": "VIDEO High Priority"
            },
            "CUSTOM": {
                "baseline_host": self.Parameters.baseline_host,
                "baseline_profile": self.Parameters.baseline_profile,
                "acceptance_host": self.Parameters.acceptance_host,
                "acceptance_profile": self.Parameters.acceptance_profile,
                "quota": self.Parameters.quota,
                "ang_quota": self.Parameters.ang_quota
            }
        }

        self.template_dir = "search/metrics_templates/video"
        self.template_name = "common.json"

        env_params = environment_params[self.Parameters.environment]

        self.baseline_host = env_params["baseline_host"]
        self.baseline_profile = env_params["baseline_profile"]
        self.acceptance_host = env_params["acceptance_host"]
        self.acceptance_profile = env_params["acceptance_profile"]
        self.quota = env_params["quota"]
        self.ang_quota = env_params["ang_quota"]

    def get_template(self):
        sdk2.svn.Arcadia.checkout("arcadia:/arc/trunk/arcadia/{}".format(self.template_dir), self.template_dir)
        eh.verify(
            os.path.exists(os.path.join(self.template_dir, self.template_name)),
            "Template '{}' doesn't exist in directory '{}'".format(self.template_name, self.template_dir)
        )

        return fu.json_load(os.path.join(self.template_dir, self.template_name))

    def on_execute(self):
        self.set_info("Launching mlm")
        self.init_params()

        launcher = metrics_launcher.MetricsLauncher(
            oauth_token=self._get_oauth_token(
                self.Parameters.vault_owner,
                self.Parameters.vault_item
            )
        )

        current_template = self.get_template()

        launch_response_info = launcher.launch_template(
            data=self.modify_template(current_template)
        )

        self.save_launch_info(launch_response_info)
        self.wait_launch_complete(launcher)

    def save_launch_info(self, info):
        self.Context.launch_info = {
            'launch_id': info['id'],
            'name': info['name']
        }

    def wait_launch_complete(self, launcher):
        self.set_info(
            'MLM url: https://mlm.yandex-team.ru/sets/{}'.format(
                self.Context.launch_info['launch_id']
            )
        )
        info = launcher.get_launch_info(self.Context.launch_info['launch_id'])
        __SLEEP_TIME = 15 * 60  # 15 minutes

        while info['status'] == u'RUNNING':
            self.set_info('Waiting until launch is done. Sleep for {} seconds.'.format(__SLEEP_TIME))
            time.sleep(__SLEEP_TIME)
            info = launcher.get_launch_info(self.Context.launch_info['launch_id'])

        if info['status'] != "COMPLETED":
            raise common.errors.TaskError("launch failed")

    def _get_oauth_token(self, vault_owner, vault_item):
        vault_data = sdk2.Vault.data(vault_owner, vault_item)
        if not vault_data:
            raise common.errors.TaskError("Failed to get vault data for {} owned by {}".format(vault_item, vault_owner))
        return vault_data

    def modify_template(self, current_template):
        launch_templates = current_template["launchTemplates"]

        for launch_template in launch_templates:
            servers = launch_template["servers"]

            for server in servers:
                host = self.baseline_host if server["baseline"] else self.acceptance_host
                profile = self.baseline_profile if server["baseline"] else self.acceptance_profile

                server["host"] = re.sub(r".*(\.yandex\.)", host + r"\1", server["host"])
                config = server["config"]
                config["quota"] = self.quota
                config["profileId"] = re.sub(r"((:?desktop|touch)__).*$", r"\1" + profile, config["profileId"])

            query_groups = launch_template["queryGroups"]

            for group in query_groups:
                group["enrichmentOptions"]["options"] = {
                    "priority": "160",
                    "angQuota": self.ang_quota
                }
                group["metrics"].append({
                    "name": "vh_share-10",
                    "externalId": "vh_share-10"
                })

        return {
            "name": 'Video robot acceptance({})'.format(self.Parameters.environment),
            "owner": self.owner,
            "launchTemplates": launch_templates,
            "description": self.Parameters.description
        }
