# -*- coding: utf-8 -*-
from sandbox import sdk2
import logging
import json
import time
import requests
import config
import sys
from collections import Counter

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.environments import PipEnvironment


class VideoRobotPriemkaSbr(sdk2.Task):
    API_URL = 'https://hitman.yandex-team.ru/api/v1/'
    UPLOAD_URL = API_URL + 'data/nirvana/upload/start/tsv'
    UPLOAD_STATUS_URL = API_URL + 'data/nirvana/upload/check?uploadTicketId={ticket_id}'
    DOWNLOAD_RESULT_URL = API_URL + 'data/nirvana/download/{task_id}'
    TASK_START_URL = API_URL + 'execution/start/{process_code}'
    EXECUTION_URL = API_URL + 'execution/{task_id}'

    class Context(sdk2.Task.Context):
        baseline_stats = {}
        acceptance_stats = {}
        baseline_percentage = 0.0
        acceptance_percentage = 0.0

    class Requirements(sdk2.Task.Requirements):
        disk_space = 100 * 1024
        environments = (PipEnvironment('yandex-yt'), PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),)

    class Parameters(sdk2.Task.Parameters):
        description = 'Video robot priemka sbr metrics calculation launcher'
        max_restarts = 1
        kill_timeout = 10 * 3600

        index_state = sdk2.parameters.String('Index state', required=True)

        environment = sdk2.parameters.String("Environment", default="PRODUCTION", choices=[(x, x) for x in ["PRODUCTION", "BETA", "CUSTOM"]])

        with environment.value["CUSTOM"]:
            baseline_host = sdk2.parameters.String('Baseline host', default=config.PRODUCTION_BASELINE_HOST)
            acceptance_host = sdk2.parameters.String('Acceptance host', default=config.PRODUCTION_ACCEPTANCE_HOST)
            target_platform = sdk2.parameters.String('Target platform', default='DESKTOP', choices=[(x, x) for x in ['DESKTOP', 'TOUCH']])

            with sdk2.parameters.Group('Vault parameters') as vault_params:
                vault_owner = sdk2.parameters.String('Vault owner', default='VIDEODEV')
                yt_vault_item = sdk2.parameters.String('YT vault item', default='yt_token')
                hitman_vault_item = sdk2.parameters.String('Hitman vault item', default='OAUTH_TOKEN')
            with sdk2.parameters.Group('Validation data') as bucket_params:
                bucket_cluster = sdk2.parameters.String('Buckets YT cluster', default=config.DEFAULT_BUCKET_PROXY)
                bucket_path_desktop = sdk2.parameters.String('Desktop bucket table', default=config.DEFAULT_DESKTOP_BUCKET)
                bucket_path_touch = sdk2.parameters.String('Touch bucket table', default=config.DEFAULT_TOUCH_BUCKET)

    def _init_params(self):
        environment_params = {
            'PRODUCTION': {
                'baseline_host': config.PRODUCTION_BASELINE_HOST,
                'acceptance_host': config.PRODUCTION_ACCEPTANCE_HOST
            },
            'BETA': {
                'baseline_host': config.BETA_BASELINE_HOST,
                'acceptance_host': config.BETA_ACCEPTANCE_HOST
            },
            'CUSTOM': {
                'baseline_host': self.Parameters.baseline_host,
                'acceptance_host': self.Parameters.acceptance_host
            }
        }
        env_params = environment_params[self.Parameters.environment]
        self.baseline_host = env_params['baseline_host']
        self.acceptance_host = env_params['acceptance_host']
        self.index_state = self.Parameters.index_state

        self.scraper_config = config.DEFAULT_SCRAPER_CONFIG

        self.requester = 'robot-video'
        self.yt_validate_desktop = self.Parameters.bucket_path_desktop
        self.yt_validate_touch = self.Parameters.bucket_path_touch
        self.bucket_proxy = self.Parameters.bucket_cluster
        self.sbr_table_proxy = config.SBR_PROXY
        self.token = sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.hitman_vault_item)
        self.yt_token = sdk2.Vault.data(self.Parameters.vault_owner, self.Parameters.yt_vault_item)
        self.target_platform = self.Parameters.target_platform

    def on_execute(self):
        self._init_params()
        table = config.VIDEOINDEX_PREFIX + self.index_state + config.SBR_STATUS_SUBPATH
        table_base = config.VIDEOSBR_FULL

        logging.info('Uploading data...')
        resource_id = self._prepare_validate_bucket(self.target_platform)

        logging.info('Launching...')
        start_response = self._start_execution(resource_id, table, self.acceptance_host)
        task_id = json.loads(start_response)['id']
        logging.info('Started acceptance task with id {}'.format(task_id))
        start_response = self._start_execution(resource_id, table_base, self.baseline_host)
        task_id_base = json.loads(start_response)['id']
        logging.info('Started baseline task with id {}'.format(task_id_base))

        logging.info('Waiting for  musca...')
        status = None
        status_base = None
        while True:
            status_response = self._do_request('get', VideoRobotPriemkaSbr.EXECUTION_URL.format(task_id=task_id))
            status = json.loads(status_response)['status']
            status_response = self._do_request('get', VideoRobotPriemkaSbr.EXECUTION_URL.format(task_id=task_id_base))
            status_base = json.loads(status_response)['status']

            logging.info('Current acceptance status: {}; baseline status: {}'.format(status, status_base))

            if status not in ['NEW', 'RUNNING', 'FIXED', 'RESUMING'] and status_base not in ['NEW', 'RUNNING', 'FIXED', 'RESUMING']:
                break
            time.sleep(300)
        if status != 'SUCCEEDED' or status_base != 'SUCCEEDED':
            raise Exception('musca failure, exit statuses: {}, {}'.format(status, status_base))

        logging.info('Fetching results...')
        results = self._do_request('get', VideoRobotPriemkaSbr.DOWNLOAD_RESULT_URL.format(task_id=task_id))

        stats, perc = self._calc_percentiles(results)

        results = self._do_request('get', VideoRobotPriemkaSbr.DOWNLOAD_RESULT_URL.format(task_id=task_id_base))
        stats_base, perc_base = self._calc_percentiles(results)

        self.Context.baseline_stats = stats_base
        self.Context.acceptance_stats = stats
        self.Context.acceptance_percentage = perc
        self.Context.baseline_percentage = perc_base

    def _start_execution(self, resource_id, table, host):
        data = {
            'requester': self.requester,
            'properties': {
                'INPUT_NIRVANA_STORED_DATA_ID': resource_id,
                'scraper_basehost': host,
                'scraper_config': self.scraper_config,
                'video_statuses_table': table
            },
            'comment': 'priemka launch'
        }
        return self._do_request('post', VideoRobotPriemkaSbr.TASK_START_URL.format(process_code=config.HITMAN_PROCESS_CODE), data=json.dumps(data))

    def _prepare_validate_bucket(self, target):
        import yt.wrapper as yt

        yt_path = None
        if target == 'DESKTOP':
            yt_path = self.yt_validate_desktop
        elif target == 'TOUCH':
            yt_path = self.yt_validate_touch
        else:
            raise Exception('Unrecognized validate bucket mode: {}'.format(target))

        yt_client = yt.YtClient(proxy=self.bucket_proxy, token=self.yt_token)
        data_string = ''
        for row in yt_client.read_table(yt_path):
            data_string += row['page_url'].decode('utf-8') + '\tDESKTOP\t' + 'RU' + '\n'

        res = self._do_request('post', VideoRobotPriemkaSbr.UPLOAD_URL, data=data_string)
        res = json.loads(res)
        while True:
            status = self._do_request('get', VideoRobotPriemkaSbr.UPLOAD_STATUS_URL.format(ticket_id=res['uploadTicketId']))
            status = status.strip('"')

            if status in ['FAILED', 'CANCELED']:
                raise Exception('upload data failed')
            if status == 'COMPLETED':
                break
            time.sleep(1)

        return res['storedDataId']

    def _calc_percentiles(self, results):
        stats = Counter()
        for line in results.split('\n'):
            status = line.split('\t')[-1]
            if len(status) <= 1:
                continue
            stats[status] += 1

        sys.path.append(svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/quality/mstand_metrics/users/pecheny/sbr/"))
        from searchable_players import count_searchable_players

        return dict(stats), count_searchable_players(stats)

    def _do_request(self, method, url, retry_nmb=0, **kwargs):
        headers = {
            'Content-type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'OAuth {}'.format(self.token)
        }
        r = requests.request(method, url, headers=headers, **kwargs)
        if r.status_code >= 400 and r.status_code < 500:
            raise Exception('request to {} failed, code {}, text: {}'.format(url, r.status_code, r.text))
        elif r.status_code >= 500 and retry_nmb < 5:
            logging.info('request to {} failed, code {}, text: {}'.format(url, r.status_code, r.text))
            logging.info('Num retries: {}'.format(retry_nmb))
            time.sleep(30)
            return self._do_request(method, url, retry_nmb=retry_nmb + 1, **kwargs)
        elif r.status_code >= 500:
            raise Exception('request to {} failed with code {}, text: {}, after {} retries'.format(url, r.status_code, r.text, retry_nmb))

        return r.text
