# -*- coding: utf-8 -*-
from sandbox import sdk2
import time
from sandbox import common
from sandbox.common.types.client import Tag


FML_LINK = r"https://fml.yandex-team.ru/view/factor/diff?right-pool-id={acceptance}&left-pool-id={baseline}"


class VideoRobotPriemkaSharpEye(sdk2.Task):
    """
        SharpEye launcher
    """
    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024  # 10 gb
        client_tags = Tag.GENERIC

    class Parameters(sdk2.Task.Parameters):
        description = "Video robot priemka sharp eye"
        max_restarts = 1  # default is 10
        kill_timeout = 4 * 3600  # 6 hours
        environment = sdk2.parameters.String("Environment", default="PRODUCTION", choices=[(x, x) for x in ["PRODUCTION"]])
        with sdk2.parameters.Group("Token parameters") as vault_params:
            vault_item = sdk2.parameters.String("Nirvana token item", default='robot_video_acc_nirvana_token')
            vault_owner = sdk2.parameters.String("Nirvana token owner", default='VIDEO-BASE-DEPLOY')

        # with environment.value["CUSTOM"]:
        #     with sdk2.parameters.Group("Sharp eye parameters") as mlm_params:
        #         baseline_host = sdk2.parameters.String("Baseline host", default="hamster.yandex.ru")
        #         acceptance_host = sdk2.parameters.String("Acceptance host", default="newdb-vid-pip.hamster.yandex.ru")
        #         workflow_id = sdk2.parameters.String("Workflow_id", default="5eb408d7-e35b-472c-8cb3-b57005e82886")
        #         baseline_scrape_block_code = sdk2.parameters.String("Baseline scrape block code", default="operation-cbc84fbe-1a21-4c53-ac7d-7ad7a7bc3f07$3")
        #         baseline_fml_block_code = sdk2.parameters.String("Baseline fml block code", default="operation-1528015836108-873$4")
        #         acceptance_scrape_block_code = sdk2.parameters.String("Acceptance scrape block code", default="operation-cbc84fbe-1a21-4c53-ac7d-7ad7a7bc3f07$1")
        #         acceptance_fml_block_code = sdk2.parameters.String("Acceptance fml block code", default="operation-1528015836108-873$5")

    class Context(sdk2.Task.Context):
        launch_info = {}

    def init_params(self):
        environment_params = {
            "PRODUCTION": {
                "baseline_host": "priemka-vid.hamster.yandex.ru",
                "acceptance_host": "newdb-vid-pip.hamster.yandex.ru",
                "workflow_id": "055c2834-b8d3-4826-90d8-3f675444f17a",
                "baseline_fml_block_code_l1": "operation-1528015836108-873$8",
                "acceptance_fml_block_code_l1": "operation-1528015836108-873$9",
                "baseline_fml_block_code_base": "operation-1528015836108-873$10",
                "acceptance_fml_block_code_base": "operation-1528015836108-873$11",
                "baseline_scrape_block_codes": ["operation-cbc84fbe-1a21-4c53-ac7d-7ad7a7bc3f07$4", "operation-cbc84fbe-1a21-4c53-ac7d-7ad7a7bc3f07$7"],
                "acceptance_scrape_block_codes": ["operation-cbc84fbe-1a21-4c53-ac7d-7ad7a7bc3f07$5", "operation-cbc84fbe-1a21-4c53-ac7d-7ad7a7bc3f07$6"]
            },
        }

        env_params = environment_params[self.Parameters.environment]

        self.baseline_host = env_params["baseline_host"]
        self.acceptance_host = env_params["acceptance_host"]
        self.workflow_id = env_params["workflow_id"]
        self.baseline_scrape_block_codes = env_params["baseline_scrape_block_codes"]
        self.baseline_fml_block_code_l1 = env_params["baseline_fml_block_code_l1"]
        self.baseline_fml_block_code_base = env_params["baseline_fml_block_code_base"]
        self.acceptance_scrape_block_codes = env_params["acceptance_scrape_block_codes"]
        self.acceptance_fml_block_code_l1 = env_params["acceptance_fml_block_code_l1"]
        self.acceptance_fml_block_code_base = env_params["acceptance_fml_block_code_base"]

    def on_execute(self):
        from sharp_eye import SharpEye

        self.set_info("Launching sharp eye")
        self.init_params()

        nirvana_token = self._get_vault_item(
            self.Parameters.vault_owner,
            self.Parameters.vault_item
        )

        self.workflow = SharpEye(self.workflow_id, nirvana_token)
        self.workflow.launch_sharp_eye(
            self.baseline_host,
            self.acceptance_host,
            baseline_scrape_block_codes=self.baseline_scrape_block_codes,
            baseline_fml_block_code_l1=self.baseline_fml_block_code_l1,
            baseline_fml_block_code_base=self.baseline_fml_block_code_base,
            acceptance_scrape_block_codes=self.acceptance_scrape_block_codes,
            acceptance_fml_block_code_l1=self.acceptance_fml_block_code_l1,
            acceptance_fml_block_code_base=self.acceptance_fml_block_code_base,
        )

        self.set_info(
            'Graph: https://nirvana.yandex-team.ru/flow/{}'.format(
                self.workflow.get_workflow_id()
            )
        )

        completion_status = self.wait_launch_complete()

        if completion_status != "success":
            raise common.errors.TaskError("Workflow failed")

        self.save_report()

    def wait_launch_complete(self):
        info = self.workflow.get_workflow_status()
        __SLEEP_TIME = 15 * 60  # 15 minutes

        while info['status'] != "completed":
            self.set_info('Waiting until launch is done. Sleep for {} seconds.'.format(__SLEEP_TIME))
            time.sleep(__SLEEP_TIME)
            info = self.workflow.get_workflow_status()

        return info['result']

    def save_report(self):
        pools = self.workflow.get_pools()
        self.Context.pools = {
            "baseline_l1": pools["baseline_l1"],
            "baseline_base": pools["baseline_base"],
            "acceptance_l1": pools["acceptance_l1"],
            "acceptance_base": pools["acceptance_base"],
        }
        self.set_info("fml compare pools l1: " + FML_LINK.format(baseline=pools["baseline_l1"], acceptance=pools["acceptance_l1"]))
        self.set_info("fml compare pools base: " + FML_LINK.format(baseline=pools["baseline_base"], acceptance=pools["acceptance_base"]))

    def _get_vault_item(self, vault_owner, vault_item):
        vault_data = sdk2.Vault.data(vault_owner, vault_item)
        if not vault_data:
            raise common.errors.TaskError("Failed to get vault data for {} owned by {}".format(vault_item, vault_owner))
        return vault_data
