# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.common import telegram
from sandbox.projects.common import metrics_launch_manager as metrics_launcher
from sandbox.projects.common.sdk_compat import task_helper as th
from sandbox.sandboxsdk.environments import PipEnvironment


class VideoRobotPriemkaSimple(sdk2.Task):
    """
        Simple video robot priemka task for 2-tier index
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 10 * 1024
        cores = 1
        environments = (PipEnvironment('yandex-yt'), PipEnvironment('yandex-yt-yson-bindings-skynet', version='0.3.32-0'),)

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Video robot acceptance for 2-tier index"
        max_restarts = 0
        kill_timeout = 20 * 3600
        mlm_task = sdk2.parameters.Task(
            "MLM execution task",
            task_type='VIDEO_ROBOT_PRIEMKA_LAUNCH_MLM',
            required=False
        )
        sharp_eye_task = sdk2.parameters.Task(
            "Sharpeye execution task",
            task_type='VIDEO_ROBOT_PRIEMKA_SHARP_EYE',
            required=False
        )
        sbr_task = sdk2.parameters.Task(
            "Sbr execution task",
            task_type='VIDEO_ROBOT_PRIEMKA_SBR',
            required=False
        )
        mlm_oauth_item = sdk2.parameters.String("OAuth vault item", default='oauth_token')
        mlm_oauth_owner = sdk2.parameters.String("OAuth vault owner", default='VIDEO-BASE-DEPLOY')
        telegram_chat_id = sdk2.parameters.String("Telegram chat id", default="-1001192710297")
        index_state = sdk2.parameters.String("Index state", default="")

    def generate_report(self):
        telegram_message = self.generate_finish_message()
        self.send_telegram_notification(telegram_message)

    def get_mlm_status(self, launch_id):
        all_borders = self.tasks_borders["mlm"]

        launcher = metrics_launcher.MetricsLauncher(
            oauth_token=sdk2.Vault.data(self.Parameters.mlm_oauth_owner, self.Parameters.mlm_oauth_item)
        )

        launches = launcher.get_launch_info(launch_id)["launches"]
        status = "GOOD"

        for launch in launches:
            if launch["name"] != "Pure":
                continue
            for query in launch["diffQueryGroups"]:
                for metric in query["metrics"]:
                    if not metric["metricId"] in all_borders:
                        continue

                    metric_borders = all_borders[metric["metricId"]]
                    if metric["signification"] == "RED" or metric["signification"] == "LIGHT_RED":
                        return "CRIT"

                    diff_percent = metric["diffPercent"]

                    if diff_percent > metric_borders["max_diff_percent"]:
                        return "CRIT"

                    if diff_percent < 0 and metric["pValue"] > metric_borders["max_pValue"]:
                        return "CRIT"

                    if diff_percent < metric_borders["min_diff_percent"]:
                        status = "WARN"

        return status

    def get_sbr_status(self, task):
        baseline_perc = self.Parameters.sbr_task.Context.baseline_percentage
        acceptance_perc = self.Parameters.sbr_task.Context.acceptance_percentage

        borders = self.tasks_borders["sbr"]
        status = "GOOD"

        if acceptance_perc - baseline_perc < borders["WARN"]:
            status = "WARN"
        if acceptance_perc - baseline_perc < borders["CRIT"]:
            status = "CRIT"

        return status

    def init_telegram_bot(self):
        telegram_token_name = "video-warden-bot-token"
        token_owner = "VIDEO-ROBOT"
        self.telegram_chat_id = self.Parameters.telegram_chat_id
        bot_token = th.vault_data(token_owner, telegram_token_name)
        self.telegram_bot = telegram.TelegramBot(bot_token)

    def mlm_telegram_notification(self):
        if self.Parameters.mlm_task is None:
            return "\nMlm launch: NOT FOUND"

        launch_id = self.Parameters.mlm_task.Context.launch_info["launch_id"]

        status = self.get_mlm_status(launch_id)
        message = "\nMlm launch https://mlm.yandex-team.ru/sets/{} ({})".format(launch_id, status)

        return message

    def sharp_eye_telegram_notification(self):
        if self.Parameters.sharp_eye_task is None:
            return "\nSharp eye l1: NOT FOUND"

        baseline_pool_l1 = self.Parameters.sharp_eye_task.Context.pools["baseline_l1"]
        baseline_pool_base = self.Parameters.sharp_eye_task.Context.pools["baseline_base"]
        acceptance_pool_l1 = self.Parameters.sharp_eye_task.Context.pools["acceptance_l1"]
        acceptance_pool_base = self.Parameters.sharp_eye_task.Context.pools["acceptance_base"]

        message = ("\nSharp eye l1: https://fml.yandex-team.ru/view/factor/diff?right-pool-id={acceptance_l1}&left-pool-id={baseline_l1}"
                   + "; base: https://fml.yandex-team.ru/view/factor/diff?right-pool-id={acceptance_base}&left-pool-id={baseline_base}")\
                                .format(baseline_l1=baseline_pool_l1, acceptance_l1=acceptance_pool_l1, baseline_base=baseline_pool_base, acceptance_base=acceptance_pool_base)

        return message

    def sbr_telegram_notification(self):
        if self.Parameters.sbr_task is None:
            return "\nSbr sandbox task: NOT FOUND"

        status = self.get_sbr_status(self.Parameters.sbr_task)
        message = '\nSbr sandbox task https://sandbox.yandex-team.ru/task/{} ({})'.format(
            self.Parameters.sbr_task.id, status)

        return message

    def send_telegram_notification(self, message):
        try:
            send_result = self.telegram_bot.send_message(self.telegram_chat_id, message, parse_mode=telegram.ParseMode.MARKDOWN)
            # self.set_info("Successful(?) send result: {}".format(send_result))
        except Exception as exc:
            self.set_info("Failed send telegram message " + message)
            self.set_info(str(exc))

    def diff_notification(self):
        import yt.wrapper as yt

        if len(self.Parameters.index_state) == 0:
            return ""

        yt_client = yt.YtClient(proxy="arnold", token=self.yt_token)
        yt_path = "//home/videoindex/full/index/{}/index_check/host_stats_err".format(self.Parameters.index_state)
        data_string = "\nDoc diff by hosts:\n"
        for row in yt_client.read_table(yt_path):
            checked_docs = row['checked_docs']
            ref_docs = row['ref_docs']
            host = row['host']
            state = row['state']
            data_string += "{}: {}, {:+d} docs ({:+2.1f}%)\n".format(
                host,
                state,
                checked_docs - ref_docs,
                (checked_docs - ref_docs) * 100.0 / max(checked_docs, ref_docs))
        return data_string

    def generate_finish_message(self):
        mlm_report = self.mlm_telegram_notification()
        sharp_eye_report = self.sharp_eye_telegram_notification()
        sbr_report = self.sbr_telegram_notification()
        diff_report = self.diff_notification()

        return "Done {} (https://sandbox.yandex-team.ru/task/{}).".format(self.Parameters.description, self.id) + mlm_report + sharp_eye_report + sbr_report + diff_report

    def init_params(self):
        self.tasks_borders = {
            "mlm": {
                "video-player-pfound-5": {
                    "max_diff_percent": 5,
                    "min_diff_percent": -0.3,
                    "max_pValue": 0.95
                }
            },
            "sbr": {
                "CRIT": -0.015,
                "WARN": -0.0025
            }
        }
        self.init_telegram_bot()
        self.yt_token = sdk2.Vault.data("VIDEODEV", "yt_token")

    def on_execute(self):
        self.init_params()
        self.generate_report()
