# -*- coding: utf-8 -*-
import requests
from sandbox.common import rest
from sandbox.common.proxy import OAuth
from sandbox.projects.common.sdk_compat import task_helper as th
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.common.types.task as ctt
import json
import os
import tempfile


class BaseTaskSettings(object):
    def __init__(self, task_type, executable_type, config_type, shardmap_resource_type, shard_number, indexing_timeout=None, sandbox_oauth_token=None, sandbox_rest_api_server=None):
        self.task_type = task_type
        self.executable_type = executable_type
        self.config_type = config_type
        self.shardmap_resource_type = shardmap_resource_type
        self.shard_number = shard_number
        self.indexing_timeout = indexing_timeout
        self.sandbox_oauth_token = sandbox_oauth_token
        self.sandbox_rest_client = rest.Client(
            base_url=str(sandbox_rest_api_server),
            auth=OAuth(sandbox_oauth_token)
        )

    def get_context(self):
        """ Create context for performance task """
        shard_names = sorted(set(self.get_shard_names()))
        ctx = {
            "new_basesearch_database_shards": ",".join(shard_names[:self.shard_number]),
            "old_basesearch_database_shards": ",".join(self.latest_stable_shards[:self.shard_number]),
            "shards_count": self.shard_number,
            "kill_timeout": 9 * 3600,
            "force_load_basesearch_database": False,
            "use_rbtorrent_transport": True,
        }
        if self.indexing_timeout:
            ctx["basesearch_indexing_timeout"] = self.indexing_timeout

        return ctx

    def get_index_name(self):
        """ Returns index name for specified CMS settings """
        shard_list = self.get_shard_names()
        index_shard_name = shard_list[0]
        index_name = "-".join(index_shard_name.split("-")[-2:]) + ".idx"
        return index_name


class PipTaskSettings(BaseTaskSettings):
    def __init__(self, nanny_service, nanny_token, **kwargs):
        super(PipTaskSettings, self).__init__(**kwargs)
        self.nanny_service = nanny_service
        self.nanny_token = nanny_token

    def load_latest_stable_shard_names(self):
        """ Save shard names from service shardmap """
        shardmap_url = None
        self.latest_stable_shards = []
        release = self.sandbox_rest_client.release.read(
            resource_type=self.shardmap_resource_type,
            type=ctt.ReleaseStatus.STABLE,
            limit=1
        )['items'][0]
        release_task_id = release['task_id']
        resource = self.sandbox_rest_client.resource.read(
            type=self.shardmap_resource_type,
            task_id=release_task_id,
            limit=1
        )['items'][0]
        shardmap_url = resource['http']['links'][0]

        for line in requests.get(shardmap_url).text.split("\n"):
            if line:
                self.latest_stable_shards.append(line.split()[1])

    def get_shard_names(self):
        shardmap_info = nanny.get_nanny_runtime_shardmap_info(
            self.nanny_service,
            self.nanny_token,
        )
        shardmap_task_id = int(shardmap_info['task_id'])
        shardmap_resource_type = shardmap_info['resource_type']
        resource = self.sandbox_rest_client.resource.read(
            type=shardmap_resource_type,
            task_id=shardmap_task_id,
            limit=1
        )['items'][0]
        shardmap_url = resource['http']['links'][0]
        shards = []
        try:
            for line in requests.get(shardmap_url).text.split("\n"):
                if line:
                    shards.append(line.split()[1])
        except AttributeError:
            pass
        return shards


class SdmsTaskSettings(BaseTaskSettings):
    def __init__(self, baseline_host, acceptance_host, **kwargs):
        super(SdmsTaskSettings, self).__init__(**kwargs)
        self.baseline_host = baseline_host
        self.acceptance_host = acceptance_host

    def get_shards(self, beta_name):
        sdms_api_url = "https://searchdms.yandex-team.ru/api/v2/beta/" + beta_name
        beta_data = json.loads(requests.get(sdms_api_url, verify=False).text)
        configs = beta_data["configuration"]["sourceConfigs"]["vid-main:VID_MMETA"]

        shardmap_torrent = configs["resources"].get("shardmap_base.map")
        if shardmap_torrent is None:
            shardmap_torrent = configs["systemResources"].get("shardmap_base.map")

        assert shardmap_torrent, "No shardmap in {}".format(beta_name)

        tempdir = tempfile.mkdtemp()
        command = 'sky get -d {path} {torrent}'.format(torrent=shardmap_torrent, path=tempdir)

        max_attempts = 5
        attempt = 0

        while attempt < max_attempts:
            if os.system(command) == 0:
                break
            attempt += 1
            if attempt == max_attempts:
                raise "Fail to download torrent {}".format(shardmap_torrent)

        shardmap_file = os.listdir(tempdir)[0]

        shards = []
        for line in open(os.path.join(tempdir, shardmap_file)).read().split("\n"):
            if line:
                shards.append(line.split()[1])

        return shards

    def get_shard_names(self):
        return self.get_shards(self.acceptance_host.split(".")[0])

    def load_latest_stable_shard_names(self):
        self.latest_stable_shards = self.get_shards(self.baseline_host.split(".")[0])


class BetaSettings:
    _ACCEPTANCE_PARAMS = {
        "task_type": "PRIEMKA_VIDEO_BASESEARCH_DATABASE",
        "executable_type": "VIDEOSEARCH_EXECUTABLE",
        "config_type": "VIDEO_SEARCH_CONFIG",
        "shardmap_resource_type": 'VIDEO_BASE_SHARDMAP',
        "shard_number": 10,
        "indexing_timeout": None
    }

    def __init__(
        self,
        sb_rest_api_server,
        sb_oauth_vault_item, sb_oauth_vault_owner,
        nanny_service,
        nanny_vault_item, nanny_vault_owner,
        baseline_host, acceptance_host,
        beta_service
    ):
        self.sb_rest_api_server = sb_rest_api_server
        self.nanny_service = nanny_service
        self.sandbox_token = th.vault_data(sb_oauth_vault_owner, sb_oauth_vault_item)
        self.nanny_token = th.vault_data(nanny_vault_owner, nanny_vault_item)
        self.baseline_host = baseline_host
        self.acceptance_host = acceptance_host
        self.beta_service = beta_service

    def videosearch_settings(self):
        if self.beta_service == "NANNY":
            return PipTaskSettings(
                nanny_service=self.nanny_service,
                nanny_token=self.nanny_token,
                sandbox_rest_api_server=self.sb_rest_api_server,
                sandbox_oauth_token=self.sandbox_token,
                **self._ACCEPTANCE_PARAMS
            )

        if self.beta_service == "SDMS":
            return SdmsTaskSettings(
                baseline_host=self.baseline_host,
                acceptance_host=self.acceptance_host,
                sandbox_rest_api_server=self.sb_rest_api_server,
                sandbox_oauth_token=self.sandbox_token,
                **self._ACCEPTANCE_PARAMS
            )
