# -*- coding: utf-8 -*-

from sandbox.common.types.task import Status
from sandbox.projects import resource_types
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
import os
import sandbox.projects.common.build.parameters as ya_make_params


class YtProxyName(SandboxStringParameter):
    description = 'YT proxy name'
    default_value = 'hahn'
    name = 'yt_proxy_name'
    required = True


class YtTokenVaultName(SandboxStringParameter):
    description = 'YT token vault name'
    default_value = 'yt_token'
    name = 'yt_token_vault_name'
    required = True


class DelayedViewTrieOpsArcadiaPath(SandboxStringParameter):
    description = 'Delayed view trie ops arcadia path'
    default_value = 'extsearch/video/quality/delayed_view/trie_ops/trie_ops'
    name = 'delayed_view_trie_ops_arcadia_path'
    required = True


def get_common_params():
    return [YtProxyName, YtTokenVaultName, DelayedViewTrieOpsArcadiaPath]


__YA_MAKE_DELAYED_VIEW_TRIE_OPS_TASK_ID = 'ya_make_delayed_view_trie_ops_task_id'
__DELAYED_VIEW_TRIE_OPS_BINARY_PATH = 'delayed_view_trie_ops_binary_path'


def __create_ya_make_subtask(task, arcadia_path, description):
    subtask_parameters = {
        ya_make_params.BuildType.name: 'Release',
        ya_make_params.ArcadiaUrl.name: Arcadia.trunk_url(),
        'targets': os.path.dirname(arcadia_path),
    }
    return task.create_subtask(
        task_type='YA_MAKE', description=description,
        input_parameters=subtask_parameters
    )


def create_ya_make_trie_ops_subtask(task):
    delayed_view_trie_ops_arcadia_path = task.ctx[DelayedViewTrieOpsArcadiaPath.name]
    subtask = __create_ya_make_subtask(
        task, delayed_view_trie_ops_arcadia_path,
        'Ya make delayed view trie ops'
    )
    task.ctx[__YA_MAKE_DELAYED_VIEW_TRIE_OPS_TASK_ID] = subtask.id
    task.wait_tasks(subtask, Status.Group.FINISH, wait_all=True)


def get_resource(task_id, resource_type):
    task = channel.sandbox.get_task(task_id)
    if task.is_finished():
        resources = channel.sandbox.list_resources(resource_type=resource_type, task_id=task_id)
        if (not resources) or len(resources) != 1:
            raise SandboxTaskFailureError(
                'Cannot find exactly one resource of type {}'
                ' for task {}'.format(resource_type, task_id)
            )
        resource = resources[0]
        if not resource.is_ready():
            raise SandboxTaskFailureError(
                'Resource of type {} from task {} '
                ' is not ready yet'.format(resource_type, task_id)
            )
        return resource
    else:
        raise SandboxTaskFailureError('Task {} has not completed correctly'.format(task_id))


def save_ya_make_trie_ops_subtask_result(task):
    subtask_id = task.ctx[__YA_MAKE_DELAYED_VIEW_TRIE_OPS_TASK_ID]
    subtask_resource = get_resource(subtask_id, resource_types.BUILD_OUTPUT)
    trie_ops_binary_path = os.path.join(
        task.sync_resource(subtask_resource),
        task.ctx[DelayedViewTrieOpsArcadiaPath.name]
    )
    task.ctx[__DELAYED_VIEW_TRIE_OPS_BINARY_PATH] = trie_ops_binary_path


def __get_from_vault(task, key):
    data = task.get_vault_data(task.owner, key)
    if data is None:
        raise SandboxTaskFailureError('Cannot extract data from vault with key {}'.format(key))
    return data


def get_yt_params(task):
    yt_proxy = task.ctx[YtProxyName.name]
    yt_token_vault_name = task.ctx[YtTokenVaultName.name]
    return (yt_proxy, __get_from_vault(task, yt_token_vault_name))


def run_in_yt_env(cmd, log_prefix, yt_token):
    os.environ['YT_TOKEN'] = yt_token
    process.run_process(cmd=cmd, log_prefix=log_prefix, wait=True, check=True)


def run_trie_ops_process(task, mode, base_table, trie_path):
    yt_proxy, yt_token = get_yt_params(task)
    trie_ops_binary_path = task.ctx[__DELAYED_VIEW_TRIE_OPS_BINARY_PATH]
    run_in_yt_env(
        [
            trie_ops_binary_path, mode, '--proxy', yt_proxy,
            '--base_table', base_table, '--trie_path', trie_path
        ],
        'delayed_view_trie_ops_{}'.format(mode), yt_token
    )


def process_stages(task, stage_funcs):
    for stage_func in stage_funcs:
        tag = '_'.join([task.type, stage_func.func_name])
        if task.ctx.get(tag, False) is False:
            task.ctx[tag] = True
            stage_func()
