# -*- coding: utf-8 -*-

import os
import tarfile

from sandbox.common.fs import untar_archive
from sandbox.common.types.resource import State
from sandbox.projects import resource_types
from sandbox.projects.common.utils import sync_resource, sync_last_stable_resource
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class VideoRecommenderSessionsToolData(sdk2.Resource):
    """Data required to run sessions tool"""


class VideoRecommenderSessionsToolConfigArcadiaPath(SandboxStringParameter):
    default_value = "/extsearch/video/quality/recommender/tools/sessions/prod/sessions.cfg"
    description = "Arcadia trunk path to sessions config"
    name = "video_recommender_sessions_tool_config_arcadia_path"
    required = True


class VideoRecommenderSessionsToolStaffConfigArcadiaPath(SandboxStringParameter):
    default_value = "/extsearch/video/quality/recommender/tools/sessions/prod/sessions_staff.cfg"
    description = "Arcadia trunk path to sessions staff config"
    name = "video_recommender_sessions_tool_staff_config_arcadia_path"
    required = True


class VideoRecommenderSessionsToolStatConfigArcadiaPath(SandboxStringParameter):
    default_value = "/extsearch/video/quality/recommender/tools/sessions/prod/sessions_stat.cfg"
    description = "Arcadia trunk path to sessions stat config"
    name = "video_recommender_sessions_tool_stat_config_arcadia_path"
    required = True


class VideoRecommenderBuildSessionsToolData(SandboxTask):
    """Builds resource with data required to run sessions tool"""

    type = "VIDEO_RECOMMENDER_BUILD_SESSIONS_TOOL_DATA"

    input_parameters = [
        VideoRecommenderSessionsToolConfigArcadiaPath,
        VideoRecommenderSessionsToolStaffConfigArcadiaPath,
        VideoRecommenderSessionsToolStatConfigArcadiaPath,
    ]

    def on_execute(self):
        result_path = "data"
        make_folder(result_path)

        from projects.geobase.Geodata4BinStable.resource import GEODATA4BIN_STABLE
        geodata_path = sync_last_stable_resource(GEODATA4BIN_STABLE)
        untar_archive(str(geodata_path), result_path)

        try:
            blockstat_path = sync_resource(
                channel.sandbox.list_resources(
                    resource_type=resource_types.YA_BLOCKSTAT_DICT,
                    status=State.READY
                )[0].id
            )
        except IndexError:
            raise SandboxTaskFailureError("Cannot find any YA_BLOCKSTAT_DICT resources")

        copy_path(str(blockstat_path), result_path)

        arcadia_paths = [
            "/yweb/webscripts/video/canonize/config/simple_owners.lst",
            "/yweb/webscripts/video/canonize/config/urlsconvert.urls.re",
            "/extsearch/video/quality/recommender/tools/sessions/data/relev_regions.vid.txt",
            "/extsearch/video/quality/recommender/tools/sessions/prod/white_list.txt",
            self.ctx[VideoRecommenderSessionsToolConfigArcadiaPath.name],
            self.ctx[VideoRecommenderSessionsToolStaffConfigArcadiaPath.name],
            self.ctx[VideoRecommenderSessionsToolStatConfigArcadiaPath.name],
        ]
        for path in arcadia_paths:
            Arcadia.export(Arcadia.trunk_url(path), result_path)

        result_archive_path = "sessions_data.tar"
        with tarfile.open(result_archive_path, "w") as result_archive:
            for filename in os.listdir(result_path):
                result_archive.add(os.path.join(result_path, filename), arcname=filename)

        result_data = self.create_resource(
            description="Video recommender sessions tool data",
            resource_path=result_archive_path,
            resource_type=VideoRecommenderSessionsToolData,
            attributes={'ttl': 'inf'}
        )

        self.mark_resource_ready(result_data)


__Task__ = VideoRecommenderBuildSessionsToolData
