# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.common import errors
import sandbox.common.types.task as ctt

from sandbox.projects import resource_types
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils
from sandbox.projects.common import apihelpers
from sandbox.projects.common.base_search_quality.response_saver_params import DefaultResponseSaverParams
from sandbox.projects.common.base_search_quality.threadPool import ProcessCount
from sandbox.projects.common.middlesearch.single_host import ReplaceBasesearches
from sandbox.projects.common.search import components as sc
from sandbox.projects.common.search import settings as media_settings
from sandbox.projects.common.search.cluster_names import C
from sandbox.projects.common.search.eventlog import eventlog

import logging
import uuid


MIDDLESEARCH_PARAMS = sc.create_middlesearch_params()


def _get_and_check_last_resource_id(resource_type):
    resource = apihelpers.get_last_resource(resource_type)
    if not resource:
        raise errors.TaskError('Cannot find resource %s' % resource_type)
    return resource.id


class VideoGetCurrentInfo(object):
    """ Get current video search info from Nanny. """

    # Nanny services that will be asked by VideoGetCurrentInfo.
    SERVICES = set()

    # Sandbox resources we are interested in.
    RESOURCE_TYPES = set()

    def __init__(self, task):
        self._task = task
        self._init_resources()

    def _init_resources(self):
        self._nanny_client = media_settings.VideoSettings.get_nanny(
            media_settings.VideoSettings.get_nanny_token(self._task)
        )

        for service_name in self.SERVICES:
            logging.info('Get attributes from a service {}'.format(service_name))
            runtime_attrs = self._nanny_client.get_service_runtime_attrs(service_name)
            try:
                sandbox_files = runtime_attrs['content']['resources']['sandbox_files']
            except KeyError:
                continue

            for desc in sandbox_files:
                if 'resource_type' in desc and 'resource_id' in desc:
                    resource_type = desc['resource_type']
                    if resource_type not in self.RESOURCE_TYPES:
                        continue

                    resource_id = desc['resource_id']

                    if not hasattr(self, resource_type):
                        logging.info('Get resource id {} for resource type {}'.format(
                            resource_id, resource_type))

                        object.__setattr__(self, resource_type, resource_id)

                    elif getattr(self, resource_type) != resource_id:
                        error_msg = 'Resource {resource_type}, id={new_id} has already existed id={old_id}'.format(
                            resource_type=resource_type,
                            new_id=resource_id, old_id=getattr(self, resource_type)
                        )
                        raise errors.TaskError(error_msg)


class VideoMiddleSearchGetCurrentInfo(VideoGetCurrentInfo):
    def __init__(self, task):
        self.SERVICES.update([
            'production_vhs_mmeta',

            # To compare versions of released resources:
            'production_vidmmeta_sas',
            'production_vidmmeta_man',
            'production_vidmmeta_vla',
        ])

        self.RESOURCE_TYPES.update([
            'VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE',
            'VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE',
        ])

        super(VideoMiddleSearchGetCurrentInfo, self).__init__(task)

        testenv_attributes = media_settings.VideoSettings.testenv_resource_attributes(
            media_settings.VideoSettings.MMETA_ID,
            media_settings.VideoSettings.PRIEMKA_SOURCE[1]
        )

        self.VIDEO_MIDDLESEARCH_DATA = utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_DATA,
            testenv_attributes[0],
            testenv_attributes[1]
        ).id

        self.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS = utils.get_and_check_last_resource_with_attribute(
            resource_types.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,
            testenv_attributes[0],
            testenv_attributes[1]
        ).id

        # See https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/PriemkaVideoBasesearchDatabase/__init__.py
        # and https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/common/priemka/database.py
        self.VIDEO_MIDDLESEARCH_INDEX = _get_and_check_last_resource_id(resource_types.VIDEO_MIDDLESEARCH_INDEX)

        # Get Mmeta hamster production config.
        self.VIDEO_MIDDLESEARCH_CONFIG = self._get_vidmmeta_config_id()

        # Just for testing.
        if self._task.Parameters.on_local:
            self.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE = _get_and_check_last_resource_id(
                resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE
            )

            self.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE = _get_and_check_last_resource_id(
                resource_types.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE
            )

    def _get_vidmmeta_instances(self):
        instances = self._nanny_client.get_service_current_instances(
            C.production_hamster_vidmmeta)['result']
        return [(i['hostname'], i['port']) for i in instances]

    def _get_vidmmeta_config_id(self):
        new_resource = sdk2.Resource['VIDEO_MIDDLESEARCH_CONFIG'](
            self._task,
            self._task.Parameters.description,
            path=uuid.uuid4(),
            test_video_recommender=True,
        )

        new_resource_data = sdk2.ResourceData(new_resource)
        output_path = str(new_resource_data.path)
        for host, port in self._get_vidmmeta_instances():
            try:
                utils.get_config_by_info_request(host, port, output_path)
                if '_CONQUISTA_' not in fu.read_file(output_path):
                    new_resource_data.ready()
                    return new_resource.id
            except errors.TaskFailure:
                pass

        raise errors.TaskError('Cannot load vidmmeta config')


class VideoRecommenderTestMiddleResources(sdk2.Task):
    """ Test middle video search resources. """

    class Parameters(sdk2.Task.Parameters):
        # Set True for testing.
        on_local = sdk2.parameters.Bool('Running on local Sandbox', default=False)

        with sdk2.parameters.Group('Shooting parameters') as shooting_parameters:
            # The queries_limit in VIDEO_RECOMMENDER_GET_MIDDLESEARCH_RESPONSES response_saver_params.
            queries_limit = sdk2.parameters.Integer(
                label=DefaultResponseSaverParams.QueriesLimitParameter.description,
                default=DefaultResponseSaverParams.QueriesLimitParameter.default_value,
                required=True
            )

            # max_fail_rate is percent of errors that we received from basesearches (SEARCH-2084).
            # It's not percent of not responded reqids! Because in this case we got an answer.
            max_fail_rate = sdk2.parameters.Float(
                label=DefaultResponseSaverParams.MaxFailRate.description + ', in [0..1]',
                default=DefaultResponseSaverParams.MaxFailRate.default_value,
                required=True
            )

            # It's percent of not responded reqids.
            max_not_responded_reqids_percent = sdk2.parameters.Float(
                label='Max percent of not responded reqids, in [0..100]',
                default=.0,
                required=True
            )

        with sdk2.parameters.Group('Verified resources') as verified_resources:
            categ_info_storage_id = sdk2.parameters.Integer(
                label='VIDEO_CATEG_INFO_STORAGE id',
                default=None
            )

            categ_info_storage_name = sdk2.parameters.String(
                label='VIDEO_CATEG_INFO_STORAGE name',
                default='categ_info_storage',
                required=True
            )

    def get_vault_data(self, vault_item_owner, vault_item_name):
        """ Added for compatibility with sandbox.projects.common.gnupg and sandboxsdk.ssh. """
        return sdk2.Vault.data(vault_item_owner, vault_item_name)

    def on_execute(self):
        with self.memoize_stage.get_middlesearch_responses:
            self._middlesearch_info = VideoMiddleSearchGetCurrentInfo(self)
            self._patch_middlesearch_resources()
            sub_task = sdk2.Task['VIDEO_RECOMMENDER_GET_MIDDLESEARCH_RESPONSES'](self, **self._gen_ctx())
            sub_task.enqueue()
            self.Context.get_middlesearch_responses_id = sub_task.id
            raise sdk2.WaitTask(sub_task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        get_middlesearch_responses = self.find(
            sdk2.Task['VIDEO_RECOMMENDER_GET_MIDDLESEARCH_RESPONSES'],
            id=self.Context.get_middlesearch_responses_id
        ).first()

        def not_responded_reqids_percent():
            return getattr(get_middlesearch_responses.Context, eventlog.NOT_RESPONDED_REQIDS_PERCENT)

        if get_middlesearch_responses.status not in ctt.Status.Group.SUCCEED:
            error_msg = 'VIDEO_RECOMMENDER_GET_MIDDLESEARCH_RESPONSES task failed.'
            raise errors.TaskFailure(error_msg)

        elif (0 < self.Parameters.max_not_responded_reqids_percent < not_responded_reqids_percent()):
            error_msg = 'VIDEO_RECOMMENDER_GET_MIDDLESEARCH_RESPONSES task failed: {} > {}.'.format(
                eventlog.NOT_RESPONDED_REQIDS_PERCENT,
                self.Parameters.max_not_responded_reqids_percent,
            )
            raise errors.TaskFailure(error_msg)

    def _gen_ctx(self):
        info = self._middlesearch_info

        ctx = {
            MIDDLESEARCH_PARAMS.Binary.name: info.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE,

            # Mmeta hamster cfg.
            MIDDLESEARCH_PARAMS.Config.name: info.VIDEO_MIDDLESEARCH_CONFIG,

            # Rearrange data.
            MIDDLESEARCH_PARAMS.Data.name: info.VIDEO_MIDDLESEARCH_DATA,

            # Rearrange index.
            MIDDLESEARCH_PARAMS.Index.name: info.VIDEO_MIDDLESEARCH_INDEX,

            # Models archive.
            MIDDLESEARCH_PARAMS.ArchiveModel.name: info.VIDEO_MIDDLE_DYNAMIC_MODELS_ARCHIVE,

            # As in web process count (do not overshoot basesearch (SEARCH-1044)).
            ProcessCount.name: 4,

            # Replace basesearches in middlesearch config.
            ReplaceBasesearches.name: False,

            # Source queries.
            DefaultResponseSaverParams.QueriesParameter.name: info.VIDEO_MIDDLESEARCH_PLAIN_TEXT_REQUESTS,

            # Recheck results N times.
            DefaultResponseSaverParams.RecheckResultsNTimesParameter.name: 1,

            # Get all factors (add gta=_RelevFactors).
            DefaultResponseSaverParams.GetAllFactorsParameter.name: True,

            # Ignore basesearch GotError.
            DefaultResponseSaverParams.IgnoreGotErrorParameter.name: True,

            # Limit number of queries to read (0 = all).
            DefaultResponseSaverParams.QueriesLimitParameter.name: self.Parameters.queries_limit,

            # Max fail rate when shooting at basesearches.
            DefaultResponseSaverParams.MaxFailRate.name: self.Parameters.max_fail_rate,
        }

        return ctx

    def _patch_middlesearch_resources(self):
        resource_to_prefix = {'VIDEO_MIDDLESEARCH_INDEX': '', 'VIDEO_MIDDLESEARCH_DATA': 'rearrange'}
        for resource_type, dst_prefix in resource_to_prefix.items():
            self._patch(resource_type, dst_prefix)

    def _patch(self, resource_type, dst_prefix):
        patch_name_to_id = {
            self.Parameters.categ_info_storage_name: self.Parameters.categ_info_storage_id
        }

        patch_file_names = set([name for name, resource_id in patch_name_to_id.items() if resource_id])
        if not patch_file_names:
            return

        old_resource_data = sdk2.ResourceData(sdk2.Resource.find(
            id=getattr(self._middlesearch_info, resource_type)
        ).first())

        new_resource = sdk2.Resource[resource_type](
            self,
            self.Parameters.description,
            path=uuid.uuid4(),
            test_video_recommender=True,
        )

        new_resource_data = sdk2.ResourceData(new_resource)

        def ignore(src, names):
            return patch_file_names if src == str(old_resource_data.path) else set()

        sdk2.paths.copytree3(
            src=str(old_resource_data.path),
            dst=str(new_resource_data.path),
            symlinks=True,
            ignore=ignore
        )

        for name, resource_id in patch_name_to_id.items():
            if resource_id:
                data = sdk2.ResourceData(sdk2.Resource.find(id=resource_id).first())
                logging.info('{resource_type}: copy {src} to {dst_prefix}/{dst}'.format(
                    resource_type=resource_type,
                    src=data.path.name,
                    dst_prefix=dst_prefix,
                    dst=name))

                if dst_prefix:
                    dst_path = new_resource_data.path.joinpath(dst_prefix)
                else:
                    dst_path = new_resource_data.path

                old_mode = dst_path.stat().st_mode
                dst_path.chmod(0o755)

                sdk2.paths.copy_path(
                    source=str(data.path),
                    destination=str(dst_path.joinpath(name))
                )

                dst_path.chmod(old_mode)

        new_resource_data.ready()
        object.__setattr__(self._middlesearch_info, resource_type, new_resource.id)
