import logging
import time

from sandbox import common
from sandbox import sdk2
import sandbox.common.types.resource as ctr

from sandbox.projects.VideoSearch.platform_helper import (
    ClusterMasterApi, ClusterMasterResponceStatus, PlatformDeployer
)

import sandbox.projects.video.quality.recommender.VideoVitrinaCuba.resource_types as rt


class PlatformDeployerVitrina(PlatformDeployer):
    _PlatformDeployer__OAUTH_VAULT_OWNER = "VITRINA-TESTENV"
    _PlatformDeployer__PLATFORM_URL = "https://qloud.yandex-team.ru/api/v1"
    _PlatformDeployer__TEMPLATE_COMPONENT_NAME = "cm-cuba"

    def __init__(self, env_id, component_name, template_component_name, variables_to_deploy={}, resources_to_deploy={}, new_component=True):
        super(PlatformDeployerVitrina, self).__init__(
            env_id, component_name, template_component_name, variables_to_deploy, resources_to_deploy, new_component
        )


class VideoVitrinaBackendTestEnv(sdk2.Task):
    __ENV_ID = "vitrina.test-vitrina.test-cuba"
    __CLUSTER_MASTER_API_URL = "https://test-cuba.common-int.yandex-team.ru"

    class Parameters(sdk2.Task.Parameters):
        CubaBundle = sdk2.parameters.Resource(
            "Bundle with binaries",
            resource_type=rt.VideoVitrinaBundle,
            state=(ctr.State.READY,),
            required=True
        )

        CubaTestDataDir = sdk2.parameters.Resource(
            "Dir with main.sh and configs",
            resource_type=rt.VideoVitrinaDataDir,
            state=(ctr.State.READY,),
            required=True
        )

        revision = sdk2.parameters.String("Revision", required=True)

    @property
    def _component_name(self):
        return "r{}-{}".format(self.Parameters.revision, self._timestamp)

    @property
    def _cluster_master_api_url(self):
        return "{}/{}".format(self.__CLUSTER_MASTER_API_URL, self._component_name)

    @property
    def _env_id(self):
        return self.__ENV_ID

    @property
    def _timestamp(self):
        if not self.Context.timestamp:
            self.Context.timestamp = str(int(time.time()))
        return self.Context.timestamp

    @sdk2.header()
    def cluster_master_link_report(self):
        return '<a href="{}/">Cluster master interface</a>'.format(self._cluster_master_api_url)

    def _get_resources(self):
        return [
            {
                "id": r.id,
                "type": r.type.name
            } for r in [self.Parameters.CubaBundle, self.Parameters.CubaTestDataDir]
        ]

    def on_execute(self):
        cluster_master_api_client = ClusterMasterApi(self._cluster_master_api_url)

        with self.memoize_stage.first_step:
            resources_to_update = self._get_resources()

            deployer = PlatformDeployerVitrina(
                self._env_id,
                self._component_name,
                "cm-cuba",
                {},
                resources_to_update
            )

            deployer.deploy()
            deployer.wait_for_deploy()

            logging.info("Environment for task {} has been formed and sent to Platform".format(self.id))

            cluster_master_api_client.invalidate_whole_path("GenerateConfig")
            cluster_master_api_client.run_whole_path("Ku")
            time.sleep(45)  # wait for statuses to update

        status = cluster_master_api_client.check_target_status("Ku")
        if status == ClusterMasterResponceStatus.WAITING:
            logging.info("Circuit testing still in action. Wait for 5 minutes")
            raise sdk2.WaitTime(5 * 60)

        if status == ClusterMasterResponceStatus.FAILURE:
            logging.info("Circuit testing failed.")
            raise common.errors.TaskFailure("Circuit testing failed")
