import os
import shutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import client as ctc
from sandbox.common.types import task as ctt
from sandbox.projects.common.build import YaPackage
from sandbox.projects.common.constants import constants
from sandbox.projects.common.build import parameters as build_params
from sandbox.projects.common.nanny import nanny

from . import packages as pkg
from . import resource_types  # noqa


class VideoRobotBuildCmPackages(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Context(sdk2.Task.Context):
        child_tasks = {}

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = build_params.ArcadiaUrl()
        arcadia_patch = build_params.ArcadiaPatch()

        with sdk2.parameters.Group('Previews CM') as previews_cm:
            with sdk2.parameters.CheckGroup('Packages') as previews_cm_packages:
                for package_name in pkg.PREVIEWS_CM:
                    previews_cm_packages.values[package_name] = package_name

        with sdk2.parameters.Group('Vpq CM') as vpq_cm:
            with sdk2.parameters.CheckGroup('Packages') as vpq_cm_packages:
                for package_name in pkg.VPQ_CM:
                    vpq_cm_packages.values[package_name] = package_name

        with sdk2.parameters.Group('Dups CM') as dups_cm:
            with sdk2.parameters.CheckGroup('Packages') as dups_cm_packages:
                for package_name in pkg.DUPS_CM:
                    dups_cm_packages.values[package_name] = package_name

        with sdk2.parameters.Group('Vicont CM') as vicont_cm:
            with sdk2.parameters.CheckGroup('Packages') as vicont_cm_packages:
                for package_name in pkg.VICONT_CM:
                    vicont_cm_packages.values[package_name] = package_name

        with sdk2.parameters.Group('Deletes CM') as deletes_cm:
            with sdk2.parameters.CheckGroup('Packages') as deletes_cm_packages:
                for package_name in pkg.DELETES_CM:
                    deletes_cm_packages.values[package_name] = package_name

        with sdk2.parameters.Group('Infra CM') as infra_cm:
            with sdk2.parameters.CheckGroup('Packages') as infra_cm_packages:
                for package_name in pkg.INFRA_CM:
                    infra_cm_packages.values[package_name] = package_name

        with sdk2.parameters.Group('Crawl CM') as crawl_cm:
            with sdk2.parameters.CheckGroup('Packages') as crawl_cm_packages:
                for package_name in pkg.CRAWL_CM:
                    crawl_cm_packages.values[package_name] = package_name

    def _build_package(self, package):
        resource_type = package['resource_type']
        build_type = package['build_type']
        build_path = package['build_path']

        description = 'Svn url for arcadia: {}\nWith patch: {}'.format(
            self.Parameters.checkout_arcadia_from_url,
            bool(self.Parameters.arcadia_patch),
        )

        if build_type == 'make':
            child_task = sdk2.Task['YA_MAKE_2'](
                self,
                description='Child of {}. Build {}'.format(self.id, resource_type),
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                arcadia_patch=self.Parameters.arcadia_patch,
                use_aapi_fuse=True,
                aapi_fallback=True,
                build_system=constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                arts=build_path,
                result_rd=description,
                result_rt=resource_type,
                result_single_file=True,
                targets=build_path,
            )

        elif build_type == 'package':
            child_task = sdk2.Task['YA_PACKAGE'](
                self,
                description='Child of {}. Build {}'.format(self.id, resource_type),
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                arcadia_patch=self.Parameters.arcadia_patch,
                use_aapi_fuse=True,
                aapi_fallback=True,
                package_resource_description={build_path: description},
                build_system=constants.SEMI_DISTBUILD_BUILD_SYSTEM,
                packages=build_path,
                package_type=YaPackage.TARBALL,
                raw_package=package.get('raw_package', True),
                resource_type=resource_type,
            )

        else:
            raise Exception('Unknown build type: {}'.format(build_type))

        child_task.Requirements.client_tags = ctc.Tag.Group.LINUX
        child_task.save().enqueue()
        assert resource_type not in self.Context.child_tasks
        self.Context.child_tasks[resource_type] = child_task.id
        self.set_info('{} build task {} started ({})'.format(resource_type, child_task.id, build_type))

    def _build(self):
        for package_name in self.Parameters.previews_cm_packages:
            self._build_package(pkg.PREVIEWS_CM[package_name])

        for package_name in self.Parameters.vpq_cm_packages:
            self._build_package(pkg.VPQ_CM[package_name])

        for package_name in self.Parameters.dups_cm_packages:
            self._build_package(pkg.DUPS_CM[package_name])

        for package_name in self.Parameters.vicont_cm_packages:
            self._build_package(pkg.VICONT_CM[package_name])

        for package_name in self.Parameters.deletes_cm_packages:
            self._build_package(pkg.DELETES_CM[package_name])

        for package_name in self.Parameters.infra_cm_packages:
            self._build_package(pkg.INFRA_CM[package_name])

        for package_name in self.Parameters.crawl_cm_packages:
            self._build_package(pkg.CRAWL_CM[package_name])

        raise sdk2.WaitTask(self.Context.child_tasks.values(), ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def _prepare_resources(self):
        for resource_type, task_id in self.Context.child_tasks.items():
            task_status = sdk2.Task[task_id].status
            if task_status != ctt.Status.SUCCESS:
                raise errors.TaskFailure('Child task {} finished with status {}'.format(task_id, task_status))

            resource = sdk2.Resource.find(task_id=task_id, type=resource_type).first()
            if not resource:
                raise errors.TaskFailure('Failed to find {} resource of {}'.format(resource_type, task_id))

            path = sdk2.ResourceData(resource).path.as_posix()
            new_path = resource_type.lower()

            if os.path.isdir(path):
                shutil.copytree(path, new_path)
            else:
                shutil.copy(path, new_path)

            new_resource = sdk2.Resource[resource_type](self, resource.description, new_path)
            sdk2.ResourceData(new_resource).ready()

        self.set_info('Done')

    def on_execute(self):
        with self.memoize_stage.build:
            self._build()

        with self.memoize_stage.prepare_resources:
            self._prepare_resources()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
