import logging
import os
import re
import shutil

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.sdk2.helpers import subprocess as sp


class ZoraclStdout(sdk2.Resource):
    pass


class ZoraclStderr(sdk2.Resource):
    pass


class ZoraclLogs(sdk2.Resource):
    pass


class VideoRobotTestTrigger(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        disk_space = 128 * 1024  # 128 GB
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup('Buildable resources mode') as mode:
            mode.values['build'] = mode.Value('Build from scratch', default=True)
            mode.values['prebuilt'] = mode.Value('Use prebuilt resource')

        with mode.value['build']:
            checkout_arcadia_from_url = sdk2.parameters.String('Checkout arcadia from url', required=True, default='arcadia:/arc/trunk/arcadia')
            arcadia_patch = sdk2.parameters.String('Arcadia patch', multiline=True)

        with mode.value['prebuilt']:
            resource_id = sdk2.parameters.String('Sandbox resource ID', required=True)

        input_urls = sdk2.parameters.String('Input urls', required=True, multiline=True)

    def _build_resources_from_scratch(self):
        with self.memoize_stage.run_subtask:
            subtask = sdk2.Task['YA_MAKE_2'](
                self,
                description='VIDEO_ROBOT_TEST_TRIGGER child task',
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                arcadia_patch=self.Parameters.arcadia_patch,
                targets='robot/zora/buildall;robot/zora/buildall/configs',
            )
            subtask.Requirements.disk_space = 128 * 1024  # 128 GB
            subtask.save().enqueue()
            self.Context.subtask_id = subtask.id
            raise sdk2.WaitTask(subtask, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        res = sdk2.Resource.find(task_id=self.Context.subtask_id, type='BUILD_OUTPUT').first()
        if res is None:
            raise Exception('Failed to get build output of subtask {}'.format(self.Context.subtask_id))
        return res

    def _get_prebuilt_resources(self):
        res = sdk2.Resource.find(id=self.Parameters.resource_id).first()
        if res is None:
            raise Exception('Failed to get resource {}'.format(self.Parameters.resource_id))
        return res

    def on_execute(self):
        res = None
        if self.Parameters.mode == 'build':
            res = self._build_resources_from_scratch()
        elif self.Parameters.mode == 'prebuilt':
            res = self._get_prebuilt_resources()
        else:
            raise Exception('Invalid mode: {}'.format(self.Parameters.mode))

        shutil.copytree(sdk2.ResourceData(res).path.as_posix(), 'arcadia')
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('chmod')) as pl:
            proc = sp.Popen(
                ['chmod', '-R', '+w', 'arcadia'],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
            )
            proc.wait()
            retcode = proc.returncode
            if retcode != 0:
                raise Exception('chmod exited with code {}'.format(retcode))

        os.chdir('arcadia/robot/zora')

        zora_runner_env = os.environ.copy()
        zora_runner_env['TERM'] = 'xterm'  # for tput
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('zora_runner')) as pl:
            proc = sp.Popen(
                [
                    'scripts/zora_runner.sh',
                    '-c', 'start',
                    '-p', '20000',  # 15000-25000 is a free port range in sandbox
                ],
                stdout=pl.stdout,
                stderr=sp.STDOUT,
                env=zora_runner_env,
            )
            proc.wait()
            retcode = proc.returncode
            if retcode != 0:
                raise Exception('zora_runner exited with code {}'.format(retcode))

        port = None
        with open('run/config/config.zora', 'r') as f:
            flag = False
            for line in f:
                if flag:
                    port = re.search(r'Port: (\d+)', line).group(1)
                    break
                if 'MessagebusServer' in line:
                    flag = True  # parse port from next line

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('zoracl')) as pl:
            with open('zoracl_stdout', 'w') as fout:
                with open('zoracl_stderr', 'w') as ferr:
                    proc = sp.Popen(
                        [
                            'run/bin/zoracl',
                            'fetch',
                            '--server', 'localhost:' + port,
                            '--source', 'video_content_parser',
                            '-F', 'p',
                            '-e',
                            '--pdfprocid', 'Video'
                        ],
                        stdin=sp.PIPE,
                        stdout=fout,
                        stderr=ferr,
                    )
                    proc.communicate(self.Parameters.input_urls)
                    retcode = proc.returncode
                    if retcode != 0:
                        raise Exception('zoracl exited with code {}'.format(retcode))

        sdk2.ResourceData(ZoraclStdout(self, 'Zoracl stdout', os.path.join(os.getcwd(), 'zoracl_stdout'))).ready()
        sdk2.ResourceData(ZoraclStderr(self, 'Zoracl stderr', os.path.join(os.getcwd(), 'zoracl_stderr'))).ready()

        os.mkdir('zoracl_logs')
        for name in os.listdir('run'):
            if name.endswith('.log'):
                shutil.copy2(os.path.join('run', name), 'zoracl_logs')
        sdk2.ResourceData(ZoraclLogs(self, 'Zoracl logs', os.path.join(os.getcwd(), 'zoracl_logs'))).ready()
