# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt
from sandbox.projects.common.nanny.client import NannyClient

_NANNY_HOST = 'http://nanny.yandex-team.ru'


class VideoRobotUpdateNannyMonitoringSettings(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024  # 5 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = "Update duty logins in nanny services"
        max_restarts = 1
        kill_timeout = 1 * 3600  # 1 hour
        with sdk2.parameters.Group("Update parameters") as update_params:
            services = sdk2.parameters.List(
                "Services to update"
            )
            with sdk2.parameters.CheckGroup("Alert methods") as alert_methods:
                alert_methods.values.email = alert_methods.Value("email", checked=True)
                alert_methods.values.sms = "sms"
                alert_methods.values.telegram = "telegram"
            align_service_info = sdk2.parameters.Bool("Force set services info", default=False)
            with align_service_info.value["True"]:
                force_responsibles = sdk2.parameters.String("Responsibles (comma separated)")
            with align_service_info.value["False"]:
                duty_order = sdk2.parameters.List(
                    "Duty order",
                    description="Comma separated list of responsibles"
                )
        with sdk2.parameters.Group("Other parameters") as other_params:
            print_debug = sdk2.parameters.Bool("Print debug info", default=False)
            vault_item = sdk2.parameters.String("Vault item for nanny oauth", default='nanny-oauth-token')
            vault_owner = sdk2.parameters.String("Vault owner", required=True)

    class Context(sdk2.Task.Context):
        test = False
        responsibles = None

    def _get_token(self, vault_owner, vault_item):
        vault_data = sdk2.Vault.data(vault_owner, vault_item)
        if not vault_data:
            raise common.errors.TaskError("Failed to get vault data for {} owned by {}".format(vault_item, vault_owner))
        return vault_data

    def get_previous_responsibles(self):
        prev = sdk2.Task.find(
            type=self.type,
            status=ctt.Status.SUCCESS
        ).order(-sdk2.Task.id).first()
        if prev:
            if self.Parameters.print_debug:
                self.set_info("Previous task found: {}".format(prev.id))
            self.set_info("Previous responsibles: {}".format(prev.Context.responsibles))
            return prev.Context.responsibles
        if self.Parameters.print_debug:
            self.set_info("No previous task found")
        return None

    def get_responsibles(self, duty_order):
        self.set_info("Getting responsibles from queue")
        new_responsibles = duty_order[0]
        prev = self.get_previous_responsibles()
        if prev:
            for idx, resp in enumerate(duty_order):
                if set(resp) == set(prev):
                    new_responsibles = duty_order[(idx + 1) % len(duty_order)]
                    break
        return new_responsibles

    def update_service(self, nanny_api, service, responsibles, alert_methods):
        # TODO:
        # - Check actual nanny settings
        self.set_info("Updating {}".format(service))
        nanny_data = nanny_api.get_service_info_attrs(service)
        if self.Parameters.print_debug:
            self.set_info("Prev data: {}".format(nanny_data))
        new_data = {}
        new_data["snapshot_id"] = nanny_data["_id"]
        new_data["comment"] = "Update monitoring settings (Sandbox task #{})".format(self.id)
        new_data["content"] = nanny_data["content"]
        new_data["content"]["monitoring_settings"]["deploy_monitoring"]["content"]["responsible"]["logins"] = responsibles
        new_data["content"]["monitoring_settings"]["deploy_monitoring"]["content"]["alert_methods"] = alert_methods
        if self.Parameters.print_debug:
            self.set_info("New data: {}".format(new_data))
        resp = nanny_api.set_service_info_attrs(service, new_data)
        if self.Parameters.print_debug:
            self.set_info("Responce: {}".format(resp))

    def on_execute(self):
        if self.Parameters.align_service_info:
            self.set_info("Getting responsibles from input")
            responsibles = list(x.strip() for x in self.Parameters.force_responsibles.split(","))
        else:
            duty_order = list(x.split(",") for x in self.Parameters.duty_order)
            duty_order = list(list(x.strip() for x in d) for d in duty_order)
            responsibles = self.get_responsibles(duty_order)
        self.set_info("New responsibles: {}".format(responsibles))
        self.set_info("Alert methods: {}".format(self.Parameters.alert_methods))
        self.Context.responsibles = responsibles
        self.set_info("Updating services")
        nanny_server = NannyClient(
            _NANNY_HOST,
            self._get_token(self.Parameters.vault_owner, self.Parameters.vault_item)
        )
        for service in self.Parameters.services:
            self.update_service(nanny_server, service, responsibles, self.Parameters.alert_methods)
