# coding: utf-8

import datetime
import logging
import json

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.videorobot import resource_types as robot_types

_REPORT_NAME = 'ExtData/YandexVideoMetrics/VideoContentSystem/SelrankMetrics'
_REPORT_SCALE = 'daily'
_DAYS = 30


class VideoRobotSelrankMetrics(SandboxTask):
    """
        Метрики новой формулы Селранка для Видео
    """
    type = 'VIDEO_ROBOT_SELRANK_METRICS'

    environment = (
        PipEnvironment('python-statface-client', '0.14.0', use_wheel=True),
    )

    def on_execute(self):
        rs = channel.sandbox.list_resources(
            resource_type=robot_types.VIDEO_SELRANK_FORMULA, limit=_DAYS, status='READY')

        if rs is None:
            self.set_info('No formulas found!')
            logging.debug('No formulas found!')
            return

        # keep formulas for _DAYS period
        now = datetime.datetime.now()
        delta = now - datetime.timedelta(days=_DAYS)
        last_resources = list(filter(lambda r: datetime.datetime.fromtimestamp(r.timestamp) >= delta, rs))

        formula_cnt = int(len(last_resources))
        self.set_info('Releases number for the last {} days: {}'.format(_DAYS, formula_cnt), do_escape=False)
        logging.debug('Here are releases for the last {} days'.format(_DAYS))
        logging.debug('\n'.join(map(lambda r: str(r.id), last_resources)))

        statface_data = self.prepare_statface_data(last_resources[0], formula_cnt)

        self.statface_upload(statface_data)

    def prepare_statface_data(self, formula_resource, formula_cnt):
        # get metrics for the last formula
        id = formula_resource.id
        path = self.sync_resource(id)
        with open(path) as formula_file:
            formula_data = json.load(formula_file)

        # prepare statface data
        formula_data[0]['fielddate'] = self.get_date(_REPORT_SCALE)
        formula_data[0]['stability'] = formula_cnt
        return formula_data

    def statface_upload(self, data):
        from statface_client import StatfaceClient
        stat_user = "robot_robot-video-acc"
        stat_pass = self.get_vault_data('robot-video-acc_statface')

        stat = StatfaceClient(stat_user, stat_pass, 'upload.stat.yandex-team.ru')
        report = stat.get_report(_REPORT_NAME)

        logging.debug('upload data: {}'.format(data[0]))
        report.upload_data(_REPORT_SCALE, data)

    def get_date(self, scale):
        fmt = {
            'daily': '%Y-%m-%d',
            'hourly': '%Y-%m-%d %H:00:00',
            'minutely': '%Y-%m-%d %H:%M:00',
        }.get(scale, None)
        if fmt is None:
            logging.fatal('Invalid scale value')
        return datetime.datetime.now().strftime(fmt)


__Task__ = VideoRobotSelrankMetrics
