import logging
import re
import subprocess
import shutil
import os
import requests
import signal
import tarfile
import time
import io

from os.path import join as pj

from sandbox import sdk2
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.vins.common.resources import VinsPackage
from sandbox.projects.resource_types import OTHER_RESOURCE


class VinsRunner(object):
    def __init__(self, vins_package_path, vins_package_name, bassist_vault_token, wizard_url, vins_port="8888", vins_resources_path=None):
        self._package_path = vins_package_path
        self._vins_package_name = vins_package_name
        self._bassist_vault_token = bassist_vault_token
        self._wizard_url = wizard_url
        self._vins_port = vins_port
        if vins_resources_path:
            self._resources_path = vins_resources_path
        else:
            self._resources_path = pj(self._package_path, 'resources')

    def get_vins_address(self):
        vins_addr = "localhost:{}".format(self._vins_port)
        return vins_addr

    def start(self, env):
        logging.info("Setting environment variables...")
        if env:
            for v in env:
                os.environ[v.split('=')[0]] = v.split('=')[-1]
        os.environ['VINS_LOG_FILE'] = pj(self._package_path, 'vins.push_client.out')

        logging.info("Starting VINS...")
        vins_stdout = pj(sdk2.paths.get_logs_folder(), '{}_stdout.log'.format(self._vins_package_name))
        work_dir = os.getcwd()
        os.chdir(self._package_path)
        os.environ['MEGAMIND_AUTH_TOKEN'] = self._bassist_vault_token
        with open(vins_stdout, 'w+') as logfile:
            vins_pid = subprocess.Popen([
                pj(self._package_path, 'bin', 'yav_wrapper'),
                '-t', 'MEGAMIND_AUTH_TOKEN',
                'sec-01cxsqmp818f86wzv3rkshpctq',
                '--',
                './bin/megamind_server',
                '--service-sources-vins-url', 'http://vins.hamster.alice.yandex.net',
                '--service-sources-req-wizard-url', self._wizard_url,
                '--service-sources-req-wizard-timeout-ms', '10000',
                '--service-sources-req-wizard-max-attempts', '1',
                '--scenarios-sources-bass-url', 'http://bass.hamster.alice.yandex.net/vins',
                '-c', pj(self._package_path, 'megamind_configs', 'dev', 'megamind.pb.txt'),
                '-p', self._vins_port
            ], preexec_fn=os.setsid, stdout=logfile, stderr=subprocess.STDOUT)

        vins_addr = self.get_vins_address()
        check_status_cmd = "curl http://{}/ping".format(vins_addr)

        try_num = 0
        while True:
            try:
                ret = subprocess.check_output([check_status_cmd], shell=True)
                logging.info("Megamind ping answer: {}".format(ret))
                if ret == 'pong':
                    break
                try_num += 1
                if try_num == 5:
                    raise SandboxTaskFailureError("VINS package {} didn't start!".format(self._vins_package_name))
            except subprocess.CalledProcessError, e:
                logging.info("Ping returned: {}".format(e.returncode))
                time.sleep(120)
                try_num += 1
                if try_num == 5:
                    raise SandboxTaskFailureError("VINS package {} didn't start!".format(self._vins_package_name))

        time.sleep(30)
        poll = vins_pid.poll()
        if poll is not None:
            logging.info("Megamind not started")
            return None

        logging.info("Megamind started")
        os.chdir(work_dir)
        return vins_pid

    def save_log(self, save_dst):
        logging.info("Saving VINS log to {}".format(save_dst))
        if os.path.exists(pj(self._package_path, 'vins.push_client.out')):
            shutil.copy(pj(self._package_path, 'vins.push_client.out'), pj(save_dst, '{}_vins.push_client.out'.format(self._vins_package_name)))

    def stop(self, vins_pid):
        logging.info("Stopping VINS...")
        os.killpg(os.getpgid(vins_pid.pid), signal.SIGTERM)
        # os.kill(vins_pid.pid, signal.SIGTERM)
        time.sleep(10)


class AliceBegemotMegamindPerfTest(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("VINS packages") as stable_params:
            test_pkg = sdk2.parameters.Resource(
                'VINS package to test',
                resource_type=VinsPackage,
                required=True
            )
            reqs_pkg = sdk2.parameters.Resource(
                'List of requests',
                resource_type=OTHER_RESOURCE,
                required=True
            )
            wizard_url = sdk2.parameters.String(
                'Req wizard url',
                default='http://alice-begemot-megamind-perf.yappy.yandex.ru/wizard',
                required=True
            )

    def get_quantile_index(self, quantile, total_count):
        if total_count == 0:
            return 0
        return int((total_count - 1) * (quantile / 100.0))

    def run_megamind_perf_test(
        self,
        package_path,
        package_name,
        test_run_description,
        log_prefix,
        bassist_vault_token,
        wizard_url,
        reqs_pack_path,
        reqs_pack_name
    ):
        logging.info("Testing VINS package: {}".format(package_name))

        env = []
        vins = VinsRunner(package_path, package_name, bassist_vault_token, wizard_url)
        vins_pid = vins.start(env)
        vins_addr = vins.get_vins_address()

        reqs_file = io.open(reqs_pack_path, mode="r", encoding="utf-8")
        lines = [line.rstrip('\n') for line in reqs_file]

        url = 'http://' + vins_addr + '/begemot'
        payload = {
            "request": {
                "event": {
                    "asr_result": [
                        {
                            "confidence": 1,
                            "utterance": "example"
                        }
                    ],
                    "type": "voice_input"
                },
                "location": {
                    "lon": 37.587937,
                    "lat": 55.733771
                }
            },
            "header": {
                "request_id": "df59ac28-f608-11e7-aaa6-74d02bba819e"
            },
            "application": {
                "timezone": "Europe/Moscow",
                "uuid": "deadbeef-040a-6765-cccc-f9a8737614dd",
                "app_version": "1.2.3",
                "client_time": "20180110T131922",
                "timestamp": "1515590362",
                "lang": "ru-RU",
                "app_id": "com.yandex.vins.shooting",
                "os_version": "5.0",
                "platform": "android"
            }
        }

        score_parts = []

        total_count = 0.0
        total_score = 0.0
        total_expected = 0.0
        for line in lines:
            total_expected += 1
            for x in range(0, 10):
                payload["request"]["event"]["asr_result"][0]["utterance"] = line
                r = requests.post(url, json=payload)

                logging.info("Testing Megamind result code: {}".format(r.status_code))
                if r.status_code == 200:
                    logging.info("Testing Megamind result: {}".format(r.text))
                    res_answer = r.text
                    res_answer = res_answer.rstrip()
                    res_parts = res_answer.split()
                    if len(res_parts) == 2:
                        logging.info("Testing Megamind result part 0: {}".format(res_parts[0]))
                        logging.info("Testing Megamind result part 1: {}".format(res_parts[1]))
                        if res_parts[0] == '200':
                            total_count += 1
                            current_score = int(res_parts[1])
                            total_score += current_score
                            score_parts.append(current_score)
                            break

        if total_count == 0 or total_expected == 0:
            return {}

        begemot_score = total_score / total_count
        success_rate = 100.0 * (total_count / total_expected)
        score_parts.sort()
        logging.info("Begemot score: {}".format(begemot_score))
        logging.info("Total expected: {}".format(total_expected))
        logging.info("Total count: {}".format(total_count))
        logging.info("Total score: {}".format(total_score))
        logging.info("Success rate: {} percents".format(success_rate))
        vins.stop(vins_pid)
        res = {
            'begemot_score': begemot_score,
            'success_rate': success_rate,
            'total_expected': total_expected,
            'total_count': total_count,
            'total_score': total_score
            }

        for q in [50, 75, 90, 95, 99, 99.9]:
            res_key = 'q' + str(q)
            res[res_key] = score_parts[self.get_quantile_index(q, total_count)]

        return res

    def prepare_vins_package(self, vins_package):
        vins_pkg_path = str(sdk2.ResourceData(vins_package).path)
        if os.path.isfile(vins_pkg_path):
            pkg_name = re.sub('\.tar\.gz', '', os.path.basename(vins_pkg_path))
            dst_path = pj(os.getcwd(), pkg_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            os.mkdir(dst_path)
            logging.info("Unpacking VINS tarball...")
            with tarfile.open(vins_pkg_path, 'r') as vins_tar:
                vins_tar.extractall(dst_path)
        else:
            pkg_name = os.path.basename(vins_pkg_path)
            dst_path = pj(os.getcwd(), pkg_name)
            if os.path.exists(dst_path):
                shutil.rmtree(dst_path)
            logging.info("Copying VINS build...")
            shutil.copytree(vins_pkg_path, dst_path)
        return dst_path, pkg_name

    def prepare_reqs(self, reqs_file):
        reqs_path = str(sdk2.ResourceData(reqs_file).path)

        pkg_name = os.path.basename(reqs_path)
        dst_path = pj(os.getcwd(), pkg_name)
        logging.info("Copying requests file...")
        shutil.copy2(reqs_path, dst_path)
        return dst_path, pkg_name

    def on_execute(self):
        self.Context.log_resource_id = self.log_resource.id

        with self.memoize_stage.run_perf_test:
            test_pkg = self.Parameters.test_pkg
            reqs_pkg = self.Parameters.reqs_pkg
            wizard_url = self.Parameters.wizard_url

            test_pkg_path, test_pkg_name = self.prepare_vins_package(test_pkg)
            reqs_pkg_path, reqs_pkg_name = self.prepare_reqs(reqs_pkg)
            self.Context.test_pkg_name = test_pkg_name
            self.Context.reqs_pkg_name = reqs_pkg_name
            bassist_vault_token = sdk2.Vault.data('robot-bassist_vault_token')
            begemot_metrics = self.run_megamind_perf_test(
                test_pkg_path,
                test_pkg_name,
                "Release candidate run #",
                "test",
                bassist_vault_token,
                wizard_url,
                reqs_pkg_path,
                reqs_pkg_name
            )
            self.Context.begemot_metrics = begemot_metrics

    @sdk2.footer()
    def test_reports(self):
        report_data = ''
        return report_data
