# -*- coding: utf-8 -*-
import logging
import os
import requests
import subprocess
import shutil

from os.path import join as pj

from sandbox import sdk2

from sandbox.common import errors as common_errors
from sandbox.common.types import task as ctt
from sandbox.projects.vins.common.resources import (
    VinsTestResult,
)
from sandbox.sandboxsdk.channel import channel


PA_TOOLS_TEST_NAME = 'run_integration_diff_tests'

# Test run parameters
EXPERIMENTS = '"mm_dont_defer_apply:1"'
PLACEHOLDERS = '"city_dialog_id:672f7477-d3f0-443d-9bd5-2487ab0b6a4c"'
RETRIES = '3'

# Megamind URLs
MEGAMIND_HAMSTER_URL = \
    'http://megamind.hamster.alice.yandex.net/speechkit/app/pa/'
MEGAMIND_RC_URL = \
    'http://megamind-rc.n.yandex-team.ru/speechkit/app/pa/'

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]


class BegemotMegamindVinsIntegrationTest(sdk2.Task):
    """Task which runs VINS integration test with custom begemot"""

    class Parameters(sdk2.Task.Parameters):
        begemot_url = sdk2.parameters.String(
            'Begemot (Megamind Shard) URL',
            default='',
            required=True
        )
        branch_name = sdk2.parameters.String(
            'Begemot (Megamind Shard) branch name',
            default='begemot-megamind-5-1',
            required=True
        )
        megamind_url_hamster = sdk2.parameters.String(
            'Megamind URL (hamster)',
            default=MEGAMIND_HAMSTER_URL,
            required=True
        )
        megamind_url_rc = sdk2.parameters.String(
            'Megamind URL (rc)',
            default=MEGAMIND_HAMSTER_URL,
            required=True
        )
        st_task = sdk2.parameters.String(
            'Startrek Task',
            default='ALICERELTEST-70',
            required=True
        )

    def generate_test_run_parameters(self,
                                     pa_tools_path=None,
                                     begemot_beta_url=None,
                                     megamind_url=None,
                                     result_filename=None):
        return [
            pa_tools_path,
            PA_TOOLS_TEST_NAME,
            '--fast',
            '--trace',
            '--experiments', EXPERIMENTS,
            '--vins-url', '"' + megamind_url + '"',
            '--req-wizard-url-v2', '"' + begemot_beta_url + '"',
            '--json-dump', result_filename,
        ]

    def generate_comment_text(self,
                              branch_name=None,
                              link_to_hamster_diff_file=None,
                              link_to_rc_diff_file=None):
        comment_text = 'Vins Integration tests for ' + \
                       branch_name + \
                       ' are done.\n' + \
                       '\n\n**Test results for Hamster (Diff):** ' + \
                       link_to_hamster_diff_file + \
                       '\n**Test results for RC (Diff):** ' + \
                       link_to_rc_diff_file + \
                       '\n\n<{Details' + \
                       '\n\nBegemot (Megamind shard) beta url: ' + \
                       self.Parameters.begemot_url + \
                       '\nMegamind Hamster url: ' + \
                       self.Parameters.megamind_url_hamster + \
                       '\nMegamind RC url: ' + \
                       self.Parameters.megamind_url_rc + \
                       '}>'
        return comment_text

    def post_comment(self,
                     robot_token=None,
                     comment_text=None,
                     ticket_target=None):
        if robot_token:
            content = {'text': comment_text}
            custom_headers = {'Content-Type': 'application/json',
                              'Accept': 'application/json',
                              'Authorization': 'OAuth ' + robot_token}
            target = 'https://st-api.yandex-team.ru/v2/issues/' \
                     + str(ticket_target) \
                     + '/comments'
            logging.debug(str(ticket_target))
            r = requests.post(target,
                              json=content,
                              headers=custom_headers)
            logging.debug(str(r.status_code))

    def execute_with_logging(self, cmd):
        logging.debug('Running {}'.format(cmd))

        s = subprocess.Popen(cmd, stderr=subprocess.PIPE, shell=True)
        exitcode = None
        while True:
            line = s.stderr.readline()
            exitcode = s.poll()

            if (not line) and (exitcode is not None):
                break
            logging.debug("pid %s: %s", os.getpid(), line.strip())

        logging.debug("pid %s: finished", os.getpid(), cmd)
        return exitcode

    def build_compare_tool(self):
        task = sdk2.Task['KOSHER_YA_MAKE'](
            self,
            description='Build compare_integration_test_runs.py',
            arts_source='alice/vins/tools/pa/compare_integration_test_runs.py',
            result_rt='VINS_TOOL_COMPARE_RESOURCE',
            result_single_file=True,
            targets='alice/bass/tools/evlog_dumper'
        )

        self.Context.compare_tool_task_id = task.id
        task.enqueue()
        return task

    def build_pa_tools(self):
        task = sdk2.Task['KOSHER_YA_MAKE'](
            self,
            description='Build pa_tools from Arcadia source',
            arts='alice/vins/tools/pa/pa_tools',
            result_rt='VINS_TOOL_PA_RESOURCE',
            result_single_file=True,
            targets='alice/vins/tools/pa'
        )

        self.Context.pa_tools_task_id = task.id
        task.enqueue()
        return task

    def run_tests_hamster(self):
        task = sdk2.Task['VINS_BEGEMOT_INTEGRATION_TEST'](
            self,
            begemot_url=self.Parameters.begemot_url,
            megamind_url=self.Parameters.megamind_url_hamster,
            description='Run VINS integration tests on Hamster',
        )

        self.Context.hamster_tests_task_id = task.id
        task.enqueue()
        return task

    def run_tests_rc(self):
        task = sdk2.Task['VINS_BEGEMOT_INTEGRATION_TEST'](
            self,
            begemot_url=self.Parameters.begemot_url,
            megamind_url=self.Parameters.megamind_url_rc,
            description='Run VINS integration tests on RC',
        )

        self.Context.rc_tests_task_id = task.id
        task.enqueue()
        return task

    def save_as_resource(self, path_to_file, resource_name, resource_filename):
        if not os.path.exists(path_to_file):
            raise common_errors.TaskFailure('File {} is not found'.format(path_to_file))
        descr = 'Created by VinsBegemotIntegrationTest'
        resource = sdk2.Resource[resource_name](self,
                                                descr,
                                                resource_filename)
        resource_data = sdk2.ResourceData(resource)
        with resource_data.path.open('wb') as dst:
            with open(path_to_file, 'rb') as src:
                shutil.copyfileobj(src, dst)
        resource_data.ready()
        return resource

    def download_resource(self, tools_resource):
        resource_path = str(sdk2.ResourceData(tools_resource).path)
        pkg_name = os.path.basename(resource_path)
        dst_path = pj(os.getcwd(), pkg_name)
        if os.path.exists(dst_path):
            shutil.rmtree(dst_path)
        logging.info('Copying {} build...'.format(pkg_name))
        shutil.copyfile(resource_path, dst_path)
        return dst_path, pkg_name

    def on_execute(self):
        with self.memoize_stage.run_ya_make:
            subtasks = []
            subtasks.append(self.run_tests_hamster())
            subtasks.append(self.run_tests_rc())
            raise sdk2.WaitTask(subtasks,
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.check_ya_make_ok:
            subtasks = list(self.find())
            if subtasks and \
               any(task.status != ctt.Status.SUCCESS for task in subtasks):
                raise common_errors.TaskFailure('Build tasks failed!')

            hamster_tests_task_id = self.Context.hamster_tests_task_id
            hamster_tests = sdk2.Resource.find(type=VinsTestResult,
                                               task_id=hamster_tests_task_id).first()

            rc_tests_task_id = self.Context.rc_tests_task_id
            rc_tests = sdk2.Resource.find(type=VinsTestResult,
                                          task_id=rc_tests_task_id).first()

            target_ticket = self.Parameters.st_task

            logging.debug('Try to post comment')
            robot_token = False
            try:
                robot_token = channel.task.get_vault_data('BASS', 'robot_bassist_startrek_oauth_token')
            except Exception as exc:
                logging.debug('Failed to get robot pass and token from vault', exc)
                robot_token = False
            if robot_token:
                branch_name = self.Parameters.branch_name
                comment_text = self.generate_comment_text(branch_name, hamster_tests.http_proxy, rc_tests.http_proxy)
                self.post_comment(str(robot_token), comment_text, target_ticket)

            logging.debug('Done.')
