# coding: utf-8


"""
This task should be executed in specific enviroment
You need to build SANDBOX_LXC_IMAGE
Parameters:

Ubuntu release: trusty
Container resource's custom description: base image for VINS
Create custom LXC image: True

Additional package repositories:
deb http://dist.yandex.ru/yandex-voicetech-trusty stable/amd64/
deb http://search.dist.yandex.ru/search/ stable/all/
deb http://search.dist.yandex.ru/search/ stable/amd64/
deb http://search.dist.yandex.ru/search/ unstable/all/
deb http://search.dist.yandex.ru/search/ unstable/amd64/

Shell script to execute during final stage:
mkdir -p /usr/local/vins
virtualenv /usr/local/vins
. /usr/local/vins/bin/activate
pip install -U pip six
curl -s 'https://github.yandex-team.ru/raw/vins/vins-dm/develop/core/requirements.txt' > requirements.txt
pip install -r requirements.txt
rm requirements.txt
echo '[filter "lfs"]
        clean = git-lfs clean -- %f
        smudge = git-lfs smudge -- %f
        required = true
' > /etc/gitconfig


find /usr/local/vins -type d -exec chmod 777 {} \;

List of packages to install during final stage, space-separated:
g++
gcc
gfortran
libatlas-base-dev
libatlas-dev
libblas-dev
liblapack-dev
libboost-python-dev
libxml2-dev
libxslt-dev
libgeobase5=5.0-52
python-dev
python-pip
python-setuptools
yandex-voice-openfst
zlib1g-dev
libgeobase5-python
python-virtualenv
git-lfs

"""
import logging

from sandbox import sdk2
import os
from sandbox.sdk2.helpers import subprocess as sp


logger = logging.getLogger(__name__)

CONTAINER_RESOURCE_ID = 319320712

SCRIPT = r"""
set -ex

PIP='pip install --disable-pip-version-check --no-cache-dir -i https://pypi.yandex-team.ru/simple/'

. /usr/local/vins/bin/activate

pip install -U pip six
hash -r

$PIP -r vins-dm/core/requirements.txt
$PIP -e vins-dm/core

export YENV_TYPE=testing
python vins-dm/tools/pa/build_skills_entity.py \
       --name {name} \
       --endpoint {endpoint} \
       --out {out_path} \
       --notify \
       --notification-timeout 200
"""


class VinsCustomEntity(sdk2.Resource):
    """ Archive with custom entity """
    ttl = 30
    auto_backup = True
    entity_name = sdk2.parameters.String("Entity name", required=True)
    env_type = sdk2.parameters.String("PA skills enviroment type", required=True)


class BuildVinsCustomEntity(sdk2.Task):
    """
    Task collects activation phrases from pa_skills api, and compile
    it to binary format

    """
    class Parameters(sdk2.Task.Parameters):
        # advanced parameter
        _container = sdk2.parameters.Container(
            "VINS environment container",
            default_value=CONTAINER_RESOURCE_ID,
            required=True
        )

        # common parameters
        kill_timeout = 3600

        # custom parameters
        name = sdk2.parameters.String("Entity name", default="skills", required=True)
        skills_endpoint = sdk2.parameters.String("PA Skills endpoint url", required=True)
        env_type = sdk2.parameters.String("PA Skills enviroment", default="production", required=True)
        vins_repo = sdk2.parameters.String(
            "VINS git repo",
            default='https://github.yandex-team.ru/vins/vins-dm.git',
            required=False,
        )
        script = sdk2.parameters.String(
            'Script', multiline=True,
            default=SCRIPT
        )

    def on_execute(self):
        entity_name = self.Parameters.name
        endpoint = self.Parameters.skills_endpoint

        path = str(self.path(entity_name + '.tar.gz'))
        resource = VinsCustomEntity(
            self, "Compiled \"%s\" entity" % entity_name, path,
            entity_name=entity_name,
            env_type=self.Parameters.env_type,
        )

        with sdk2.helpers.ProcessLog(self, logger='git') as pl:
            cmd = ['/usr/bin/git', 'lfs', 'clone', self.Parameters.vins_repo]
            sp.check_call(cmd, stdout=pl.stdout, stderr=sp.STDOUT, shell=False)

        with sdk2.helpers.ProcessLog(self, logger='run_script') as pl:
            script = self.Parameters.script.format(
                name=entity_name,
                endpoint=endpoint,
                out_path=path,
            )
            proc = sp.Popen(script, stdout=sp.PIPE, stderr=pl.stdout, shell=True,
                            env=dict(os.environ, PYTHONPATH=""))
            for line in iter(proc.stdout.readline, ''):
                if line.strip() == path:
                    # copmpilation have finished
                    data = sdk2.ResourceData(resource)
                    data.ready()
                    break

            retcode = proc.wait()
            if retcode != 0:
                raise sp.CalledProcessError(retcode, script)
