import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.build import parameters as build_params
import sandbox.common.types.task as ctt

logger = logging.getLogger(__name__)


class GenerateBegemotMegamindBeta(sdk2.Task):

    class Requirements(sdk2.Requirements):
        tasks_resource = sdk2.Task.Requirements.tasks_resource()

    class Parameters(sdk2.Parameters):
        kill_timeout = 3600

        svn_url = sdk2.parameters.String(
            'Svn url for arcadia',
            required=True,
            default='arcadia:/arc/trunk/arcadia',
        )

        apply_patch = sdk2.parameters.String(
            'Apply patch (link to Arcanum review formatted as `arc:id`, diff file rbtorrent,'
            'paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
            required=False,
        )

        yappy_component_name = sdk2.parameters.String(
            'yappy_component_name',
            required=True,
            default='megabegemot',
        )

        yappy_beta_conf_type = sdk2.parameters.String(
            'yappy_beta_conf_type',
            required=True,
            default='dev-beta',
        )

        yappy_beta_name = sdk2.parameters.String(
            'yappy_beta_name',
            required=True,
        )

    def _check_subtask(self, task_id):
        logging.info('Check task {}'.format(task_id))
        task = sdk2.Task.find(id=task_id).first()

        if task.status != 'SUCCESS':
            raise TaskFailure('Subtask id:{} finished with status:{}'.format(task.id, task.status))

    def on_execute(self):
        with self.memoize_stage.build_resources:
            params = {
                build_params.UseArcadiaApiFuse.name: True,
                build_params.ArcadiaPatch.name: self.Parameters.apply_patch,
                'begemot_resources': 'BegemotConfig MegamindExecutable Bstr Megamind NannyConfigs',
                'checkout_arcadia_from_url': self.Parameters.svn_url,
            }
            build_task_class = sdk2.Task["RELEASE_BEGEMOT_MEGAMIND_RESOURCES"]
            build_task = build_task_class(self, description="Created from SDK2-task", **params).enqueue()

            self.Context.build_task = build_task.id
            raise sdk2.WaitTask(build_task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, timeout=60 * 60 * 3)

        with self.memoize_stage.run_yappy:
            logging.info("Check task %s" % self.Context.build_task)
            build_task = sdk2.Task.find(id=self.Context.build_task).first()
            self._check_subtask(self.Context.build_task)

            build_task_ctx = build_task.Context
            builded_resources = {
                "args": build_task_ctx.BEGEMOT_ARGUMENTS_PARSER,
                "begemot": build_task_ctx.BEGEMOT_MEGAMIND_EXECUTABLE,
                "bstr": build_task_ctx.BEGEMOT_BSTR,
                "bstr_caller": build_task_ctx.BEGEMOT_BSTR_CALLER,
                "eventlog_uploader": build_task_ctx.BEGEMOT_EVLOG_UPLOADER,
                "evlogdump": build_task_ctx.BEGEMOT_EVLOGDUMP,
                "fast_build_config.json": build_task_ctx.BEGEMOT_FAST_BUILD_CONFIG_MEGAMIND,
                "fast_build_data_downloader": build_task_ctx.BEGEMOT_FAST_BUILD_DOWNLOADER,
                "fast_data_callback": build_task_ctx.BEGEMOT_FAST_DATA_CALLBACK,
                "instancectl.conf": build_task_ctx.BEGEMOT_INSTANCECTL_CONF,
                "worker.cfg": build_task_ctx.BEGEMOT_CONFIG,
            }
            params = {
                'beta_conf_type': self.Parameters.yappy_beta_conf_type,
                'beta_name_source': 'STRING',
                'component_name': self.Parameters.yappy_component_name,
                'component_resources': builded_resources,
                'debug_mode': False,
                'force_start_beta': True,
                'get_beta_names_from_state': False,
                'max_wait_num': 5,
                'patch_name': self.Parameters.yappy_beta_name,
                'release_number': 0,
                'update_beta_mode': 'APPEND',
            }
            logger.info(params)

            gen_beta_class = sdk2.Task["GENERATE_YAPPY_BETA"]
            gen_beta = gen_beta_class(self, description="Generate beta", **params).enqueue()

            self.Context.gen_beta = gen_beta.id
            raise sdk2.WaitTask(gen_beta, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, timeout=60 * 60 * 3)

        with self.memoize_stage.check_beta:
            self._check_subtask(self.Context.gen_beta)
