import json
import logging
import os

from sandbox import sdk2
from sandbox.projects.vins.common.resources import (MegamindResponses, MegamindDiff, MegamindShooterBinary)
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.projects.vins.common.constants as consts


class MegamindCompareResponses(sdk2.Task):
    """ Task for megamind responses comparison """

    class Requirements(sdk2.Task.Requirements):
        # Warning - disable this on dev machines
        client_tags = consts.CLIENT_TAGS

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Responses') as responses_group:
            task1_resource = sdk2.parameters.Resource(
                'Resource with OLD megamind responses',
                resource_type=MegamindResponses,
                required=True
            )
            task2_resource = sdk2.parameters.Resource(
                'Resource with NEW megamind responses',
                resource_type=MegamindResponses,
                required=True
            )
        with sdk2.parameters.Group('Parameters') as params_group:
            shooter_binary_resource = sdk2.parameters.Resource(
                'Megamind shooter resource',
                resource_type=MegamindShooterBinary,
                required=True
            )
            diffs_per_file = sdk2.parameters.Integer(
                'Diffs per one HTML file',
                default=20,
                required=True
            )
            thread_count = sdk2.parameters.Integer(
                'Working threads count',
                default=10,
                required=True
            )
        kill_timeout = 1 * 60 * 60  # 1 hour

    def on_execute(self):
        task1_resource_data = sdk2.ResourceData(self.Parameters.task1_resource)
        task2_resource_data = sdk2.ResourceData(self.Parameters.task2_resource)
        shooter_binary_resource_data = sdk2.ResourceData(self.Parameters.shooter_binary_resource)

        megamind_diff = MegamindDiff(
            task=self,
            description='Difference in megamind responses',
            path='diff_dir'
        )
        data = sdk2.ResourceData(megamind_diff)
        os.mkdir(str(data.path))

        out_path = os.path.join(sdk2.paths.get_logs_folder(), "shooter.out")
        err_path = os.path.join(sdk2.paths.get_logs_folder(), "shooter.err")
        stats_path = os.path.join(sdk2.paths.get_logs_folder(), "stats.json")

        cmd = [
            str(shooter_binary_resource_data.path), 'diff',
            '--old-path', str(task1_resource_data.path),
            '--new-path', str(task2_resource_data.path),
            '--output-path', str(data.path),
            '--threads', str(self.Parameters.thread_count),
            '--diffs-per-file', str(self.Parameters.diffs_per_file),
            '--stats-path', stats_path
        ]

        with open(out_path, "w") as out, open(err_path, "w") as err:
            sp.Popen(
                cmd,
                stdout=out,
                stderr=err,
                stdin=None
            ).wait()

        data.ready()

        with open(stats_path, "r") as stats_file:
            stats = json.load(stats_file)
        diff_count = stats["responses_diffs_count"] or 0
        logging.debug('Got %s diffs', diff_count)

        self.Context.has_diff = diff_count > 17

    @sdk2.header()
    def header(self):
        if self.Context.has_diff:
            return 'There are diffs! Watch diff resource'
        return 'No diffs found'
