# coding: utf-8

import json
import logging
import requests
import urllib

import jinja2

from sandbox import sdk2, common


logger = logging.getLogger(__name__)


def fetch(url):
    response = requests.get(url, verify=False)
    if not response.ok:
        logger.error('Failed to fetch %s: %s %s', url, response.status_code, response.reason)
        return None
    return response


def impact(x):
    return x['lines_count'] - x['lines_covered']


def impact_cover(item, revision):
    return (impact(item) * 100.) / revision['lines_count']


def impact_remove(item, revision):
    lines_count = revision['lines_count'] - item['lines_count']
    old_lines_coverage = (revision['lines_covered'] * 100.) / revision['lines_count']
    if lines_count == 0:
        return 0
    return ((revision['lines_covered'] - item['lines_covered']) * 100.) / lines_count - old_lines_coverage


EMAIL_FOOTER_TEMPLATE = """
<br>
<p>
    Sandbox Task: <a href="https://sandbox.yandex-team.ru/task/{{id}}">{{id}}</a>
</p>
"""


TEMPLATE = """
{% macro colored(perc) %}
    rgb({{ (100 - perc) * 2.55 }}, {{ perc * 2 }}, 20)
{% endmacro %}

{% macro colored2(v, a, b) %}
    {# There is no filter max in sandbox env #}
    {% set perc = (v - a) / (b - a) * 100 %}
    {% if perc > 100 %}
        {% set perc = 100 %}
    {% elif perc < 0 %}
        {% set perc = 0 %}
    {% endif %}
    {{ colored(perc) }}
{% endmacro %}

{% macro signed_percent(v) %}
    {% if v < 0 %}
        −{{ "%0.2f" % -v }}%
    {% else %}
        +{{ "%0.2f" % v }}%
    {% endif %}
{% endmacro %}

<h3>
    C++ Coverage (<a href="https://a.yandex-team.ru/arc/commit/{{revision.revision}}">r{{ revision.revision }}</a>):
    <span style="color:{{ colored(revision.coverage) }};">
        {{revision.coverage}}%
    </span>
</h3>
<table>
    <tr style="text-align: left;">
        <th style="padding: 2px 3px; padding-left: 0;">Filename</th>
        <th style="padding: 2px 3px;">Lines count</th>
        <th style="padding: 2px 3px;">Coverage</th>
        <th style="padding: 2px 3px;">Impact (Cover / Remove)</th>
    </tr>
    {% for item in report %}
        <tr>
            <td style="padding: 2px 3px; padding-left: 0;">
                <a href="https://a.yandex-team.ru/arc/trunk/arcadia/{{item.filename}}?rev={{revision.revision}}&coverage=true">
                    {{ item.filename }}
                </a>
            </td>
            <td style="padding: 2px 3px; text-align: right;">{{ item.lines_count }}</td>
            <td style="padding: 2px 3px; color:{{ colored(item.lines_coverage) }}; text-align: right;">
                {{ "%0.2f" % item.lines_coverage }}%
            </td>
            <td style="padding: 2px 3px; text-align: right;">
                <span style="color:{{ colored2(-item.impact_cover, -1.5, 2.0) }};">
                    {{ signed_percent(item.impact_cover) }}
                </span>
                /
                <span style="color:{{ colored2(item.impact_remove if item.impact_remove > 0 else -1.5, -1.5, 1.5) }};">
                    {{ signed_percent(item.impact_remove) }}
                </span>
            </td>
        </tr>
    {% endfor %}
</table>
"""


class MegamindCoverage(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        path = sdk2.parameters.String('Path', default='alice/megamind')
        top = sdk2.parameters.Integer('Top', default=30, description='Top N uncovered files')
        fallback_revisions = sdk2.parameters.Integer('Fallback revisions',
                                                     default=2, description='Try fallback to N previous revisions')
        recipients = sdk2.parameters.List('Send email notifications to', default=['g-kostin'])

    class Context(sdk2.Task.Context):
        revision = None
        total_coverage = 0.0
        report = ''

    def on_execute(self):
        base_url = 'https://ci.yandex-team.ru/api/v1.0/coverage/'

        url = base_url + '?' + urllib.urlencode({
            'path': self.Parameters.path,
            'order': '-revision',
            'limit': self.Parameters.fallback_revisions,  # try to fallback
        })

        revisions = [self.Context.revision] if self.Context.revision else []

        if not revisions:
            response = fetch(url)
            if not response:
                raise common.errors.TaskFailure('Failed to fetch coverage revisions')
            result = json.loads(response.content)
            for item in result.get('items', {}):
                if item['type'] == 'C++ unit tests':
                    revisions = [{
                        'revision': point['revision'],
                        'coverage': point['coverage'],
                        'lines_count': point['lines_count'],
                        'lines_covered': point['lines_covered'],
                    } for point in item['points']]
                    break
            if not revisions:
                raise common.errors.TaskFailure('No c++ coverage found')

        response = None
        for revision in revisions:
            self.Context.revision = revision

            url = base_url + 'by_file?' + urllib.urlencode({
                'path': self.Parameters.path,
                'revision': revision['revision'],
            })

            response = fetch(url)
            if response:
                break
        if not response:
            raise common.errors.TaskFailure('Failed to fetch file list')
        result = json.loads(response.content)
        items = sorted(result.get('items', []), key=impact, reverse=True)

        self.Context.report = [{
            'filename': item['filename'],
            'lines_count': item['lines_count'],
            'lines_coverage': item['lines_coverage'],
            'impact_cover': impact_cover(item, self.Context.revision),
            'impact_remove': impact_remove(item, self.Context.revision),
        } for item in items[:self.Parameters.top]]

        if self.Parameters.recipients:
            self.server.notification(
                subject="Megamind Coverage",
                body=self.render_report(add_footer=True),
                recipients=self.Parameters.recipients,
                transport=common.types.notification.Transport.EMAIL,
                type=common.types.notification.Type.HTML,
                charset=common.types.notification.Charset.UTF,
                task_id=self.id,
                urgent=False
            )

    @sdk2.header()
    def render_report(self, add_footer=False):
        template = jinja2.Template(TEMPLATE)
        template.environment.trim_blocks = True
        report = template.render({
            'report': self.Context.report,
            'revision': self.Context.revision,
        })
        if add_footer:
            template = jinja2.Template(EMAIL_FOOTER_TEMPLATE)
            template.environment.trim_blocks = True
            footer = template.render({'id': self.id})
            report = '\n'.join([report, footer])
        return report
