# coding: utf-8

import logging
import os
import re
import urllib

import attr
import requests

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia

logger = logging.getLogger(__name__)

DEFAULT_PERCENTILES = '50,75,90,95,99,99.9'
DEFAULT_SCENARIOS_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/alice/megamind/configs/production/scenarios'
DEFAULT_TOKEN_NAME = 'solomon.token'
DEFAULT_TOKEN_OWNER = 'robot-voiceint'

DASHBOARD_API_URL = 'https://solomon.yandex.net/api/v2/projects/megamind/dashboards/{dashboard_id}'
SCENARIOS_PATH = 'scenarios'
SCENARIO_NAME_RE = re.compile(r'^MetricsName\s*:\s*"(?P<scenario_name>[^"]+)"')


@attr.s()
class DashboardParameter(object):
    name = attr.ib()
    value = attr.ib()


@attr.s()
class DashboardRow(object):
    panels = attr.ib(default=attr.Factory(list))

    def add_panel(self, panel):
        self.panels.append(panel)
        return self


@attr.s()
class DashboardPanel(object):
    title = attr.ib()
    url = attr.ib()
    subtitle = attr.ib(default=attr.Factory(unicode))
    rowspan = attr.ib(default=1)
    colspan = attr.ib(default=1)
    type = attr.ib(default=attr.Factory(lambda: 'IFRAME'))


@attr.s()
class Dashboard(object):
    id = attr.ib()
    projectId = attr.ib()
    name = attr.ib()
    description = attr.ib()
    heightMultiplier = attr.ib(default=2)
    parameters = attr.ib(default=attr.Factory(list))
    rows = attr.ib(default=attr.Factory(list))
    version = attr.ib(default=1)

    def add_row(self, row):
        self.rows.append(row)
        return self

    def add_parameter(self, parameter):
        self.parameters.append(parameter)
        return self

    def to_dict(self):
        return attr.asdict(self)

    def upload(self, token):
        url = DASHBOARD_API_URL.format(dashboard_id=self.id)
        headers = {
            'Authorization': 'OAuth {}'.format(token)
        }
        info = requests.get(url, headers=headers)
        info.raise_for_status()
        self.version = info.json().get('version')
        requests.put(url, headers=headers, json=self.to_dict()).raise_for_status()
        logger.info('Dashboard "%s" (id:%s) has been updated', self.name, self.id)


def load_scenario_names(scenarios_url):
    Arcadia.checkout(url=scenarios_url, path=SCENARIOS_PATH)
    scenario_names = []
    for filename in sorted(os.listdir(SCENARIOS_PATH)):
        if not filename.endswith('pb.txt'):
            continue
        with open(os.path.join(SCENARIOS_PATH, filename), 'r') as config:
            for line in config:
                match = SCENARIO_NAME_RE.match(line)
                if not match:
                    continue
                scenario_name = match.groupdict().get('scenario_name')
                if scenario_name:
                    scenario_names.append(scenario_name)
                    break
    return scenario_names


def make_panel_url(**kwargs):
    return '/?' + urllib.urlencode(kwargs).replace('%7B', '{').replace('%7D', '}')


def make_dashboard(name):
    return Dashboard(
        id=name,
        projectId='megamind',
        name=name,
        description='',
        parameters=[
            DashboardParameter(name='sensor', value='SOURCES-megamind_scenarios_{{request_type}}_stage-SCENARIO_{{scenario_name}}_{{request_type_upper}}-{{dashboard_type}}'),
            DashboardParameter(name='request_type', value='run'),
            DashboardParameter(name='dashboard_type', value='ResponseTimesMcsHistogram'),
            DashboardParameter(name='host', value='cluster'),
            DashboardParameter(name='stack', value='true'),
        ],
    )


def add_per_scenario_panel(dashboard, scenario_names, url_creator, common_panel_creator):
    for scenario_name in scenario_names:
        dashboard.add_row(
            DashboardRow().add_panel(
                DashboardPanel(
                    title='{} – {}'.format(scenario_name, '{{request_type}}'),
                    subtitle='{{sensor}}',
                    url=url_creator(scenario_name, common_panel_creator)
                )
            )
        )


def get_common_panel_params(scenario_name):
    return {
        'sensor': 'SOURCES-megamind_scenarios_{{{{request_type}}}}_stage-SCENARIO_{scenario_name}_{{{{request_type_upper}}}}-{{{{dashboard_type}}}}'.format(scenario_name=scenario_name),
        'host': '{{host}}',
        'graph': 'auto',
        'project': 'megamind',
        'cluster': 'prod',
        'service': 'apphost_alice_vertical',
        'stack': '{{stack}}',
        'transform': '{{transform}}',
    }


def make_percentile_panel_url(scenario_name, common_panel_creator):
    return make_panel_url(
        overLinesTransform='WEIGHTED_PERCENTILE',
        percentiles=DEFAULT_PERCENTILES,
        **common_panel_creator(scenario_name)
    )


def make_rate_panel_url(scenario_name, common_panel_function):
    return make_panel_url(**common_panel_function(scenario_name))


def make_percentile_dashboard(scenario_names):
    dashboard = make_dashboard('per_protocol_scenario.percentiles')
    add_per_scenario_panel(dashboard, scenario_names, make_percentile_panel_url, get_common_panel_params)
    return dashboard


def make_rate_dashboard(scenario_names):
    dashboard = make_dashboard('per_protocol_scenario.rate')
    add_per_scenario_panel(dashboard, scenario_names, make_rate_panel_url, get_common_panel_params)
    return dashboard


class MegamindDashboardCreator(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        scenarios_url = sdk2.parameters.String('Scenarios ArcadiaURL', default=DEFAULT_SCENARIOS_URL)
        solomon_token_owner = sdk2.parameters.String('Solomon Vault token owner', default=DEFAULT_TOKEN_OWNER)
        solomon_token_name = sdk2.parameters.String('Solomon Vault token name', default=DEFAULT_TOKEN_NAME)

    def on_execute(self):
        scenario_names = load_scenario_names(self.Parameters.scenarios_url)
        token = sdk2.Vault.data(self.Parameters.solomon_token_owner, self.Parameters.solomon_token_name)
        logger.info('Loaded %d scenarios: [%s]', len(scenario_names), ', '.join(scenario_names))
        make_percentile_dashboard(scenario_names).upload(token)
        make_rate_dashboard(scenario_names).upload(token)
