from os.path import join
import json
import logging
import os
import shutil

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.paths import get_logs_folder
import sandbox.common.types.task as ctt

from sandbox.projects.vins.common.resources import (VinsPackage, MegamindRequests, MegamindResponses, MegamindShooterBinary)
import sandbox.projects.vins.common.constants as consts


class MegamindGetResponsesBeta(sdk2.Task):
    ''' Get Megamind responses on various inputs (newer version) '''

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('PyYAML')
        ]

        # Warning - disable this on dev machines
        client_tags = consts.CLIENT_TAGS
        cores = consts.CORES
        dns = consts.DNS
        disk_space = consts.DISK_SPACE
        ram = consts.RAM
        privileged = consts.PRIVILEGED

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Packages') as packages_parameters:
            vins_package = sdk2.parameters.Resource('VINS package', resource_type=VinsPackage, required=True)
            megamind_requests = sdk2.parameters.Resource('Megamind requests', resource_type=MegamindRequests, required=True)
            shooter_binary = sdk2.parameters.Resource('Megamind shooter resource', resource_type=MegamindShooterBinary, required=True)

        with sdk2.parameters.Group('Secrets') as secrets_parameters:
            bass_vault_owner = sdk2.parameters.String('BASS vault owner', default='BASS', required=True)
            bass_vault_name = sdk2.parameters.String('BASS vault name', default='robot-bassist_vault_token', required=True)
            test_user_vault_owner = sdk2.parameters.String('Test User vault owner', default='BASS', required=True)
            test_user_vault_name = sdk2.parameters.String('Test User vault name', default='alice-diff-test-token', required=True)

        with sdk2.parameters.Group('Config') as config_parameters:
            # non-YDB config
            handwritten_config = sdk2.parameters.JSON('Handwritten config to use')

            # config from YDB
            ydb_token_vault_owner = sdk2.parameters.String('YDB Token vault owner', default='BASS')
            ydb_token_vault_name = sdk2.parameters.String('YDB Token vault name', default='robot_bassist_ydb_token')
            config_endpoint = sdk2.parameters.String('Config YDB endpoint', default='ydb-ru.yandex.net:2135')
            config_database = sdk2.parameters.String('Config YDB database', default='/ru/home/sparkle/mydb')
            config_table = sdk2.parameters.String('Config YDB table', default='configs')
            config_id = sdk2.parameters.String('Config YDB id in table', default='get_responses')

        with sdk2.parameters.Group('Misc') as misc_parameters:
            joker_semaphore_name = sdk2.parameters.String('Joker semaphore name', default='JOKER_CONNECTIONS')

        kill_timeout = 2 * 60 * 60  # 2 hour

    def on_enqueue(self):
        if self.Parameters.joker_semaphore_name:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=self.Parameters.joker_semaphore_name)
                ],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            )

    def prepare_vins_package(self, vins_pkg_path):
        pkg_name = os.path.basename(vins_pkg_path)
        dst_path = join(os.getcwd(), pkg_name)
        if os.path.exists(dst_path):
            shutil.rmtree(dst_path)
        logging.info('Copying VINS build...')
        shutil.copytree(vins_pkg_path, dst_path)
        logging.info('Copying VINS build ended')
        return dst_path

    def on_execute(self):
        bass_token = str(sdk2.Vault.data(self.Parameters.bass_vault_owner, self.Parameters.bass_vault_name))
        test_user_token = str(sdk2.Vault.data(self.Parameters.test_user_vault_owner, self.Parameters.test_user_vault_name))

        # load resources
        vins_package_dir = self.prepare_vins_package(str(sdk2.ResourceData(self.Parameters.vins_package).path))
        megamind_requests_dir = str(sdk2.ResourceData(self.Parameters.megamind_requests).path)
        shooter_binary = str(sdk2.ResourceData(self.Parameters.shooter_binary).path)

        sp.Popen(['chmod', '-R', '755', vins_package_dir]).wait()

        # download config
        config_path = join(get_logs_folder(), "config.json")
        if self.Parameters.handwritten_config:
            with open(config_path, 'w') as cf:
                cf.write(self.Parameters.handwritten_config)
        else:
            ydb_token = str(sdk2.Vault.data(self.Parameters.ydb_token_vault_owner, self.Parameters.ydb_token_vault_name))
            out_path = join(get_logs_folder(), "shooter_config.out")
            err_path = join(get_logs_folder(), "shooter_config.err")

            with open(out_path, 'w') as out, open(err_path, 'w') as err:
                sp.Popen(
                    [
                        shooter_binary, 'download-config',
                        '--endpoint', self.Parameters.config_endpoint,
                        '--database', self.Parameters.config_database,
                        '--ydb-token', ydb_token,
                        '--table', self.Parameters.config_table,
                        '--id', self.Parameters.config_id,
                        '--save-path', config_path
                    ],
                    stdout=out, stderr=err, stdin=None
                ).wait()

        # fix config and launch shooter
        results_dir = join(os.getcwd(), 'results')
        resource = sdk2.ResourceData(MegamindResponses(
            task=self,
            description='Responses from MegamindGetResponsesBeta task #{}'.format(self.id),
            path=results_dir
        ))

        with open(config_path, 'r') as cf:
            config = json.load(cf)
        config['requests_path'] = megamind_requests_dir

        new_runs_settings = config.get('runs_settings', [])
        if not new_runs_settings:
            new_runs_settings = [
                {
                    "package_path": vins_package_dir,
                    "logs_path": get_logs_folder(),
                    "responses_path": results_dir,
                }
            ]
        else:
            for rs in new_runs_settings:
                rs['package_path'] = vins_package_dir
                rs['logs_path'] = get_logs_folder()
                rs['responses_path'] = results_dir
        config['runs_settings'] = new_runs_settings

        fixed_config_path = join(get_logs_folder(), 'fixed_config.json')
        with open(fixed_config_path, 'w') as cf:
            json.dump(config, cf)

        out_path = join(get_logs_folder(), "shooter.out")
        err_path = join(get_logs_folder(), "shooter.err")

        with open(out_path, 'w') as out, open(err_path, 'w') as err:
            sp.Popen(
                [
                    shooter_binary, 'run', fixed_config_path,
                    '--yav-token', bass_token,
                    '--test-user-token', test_user_token
                ],
                stdout=out, stderr=err, stdin=None
            ).wait()

        resource.ready()
